/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

@file:Suppress("ClassName")

package com.bytedance.ultimate.inflater.plugin.arsc

import java.lang.reflect.Field
import kotlin.reflect.KClass

/**
 * Created by chentao.joe on 2022/7/28
 * @author chentao.joe@bytedance.com
 */
@Target(AnnotationTarget.FIELD)
annotation class unit_8_t

@Target(AnnotationTarget.FIELD)
annotation class unit_16_t

@Target(AnnotationTarget.FIELD)
annotation class unit_32_t

@Target(AnnotationTarget.FIELD)
annotation class unit_size_t_group

val Annotation.isUnitAnnotation: Boolean
    get() {
        return when (this) {
            is unit_8_t,
            is unit_16_t,
            is unit_32_t,
            is unit_size_t_group -> true

            else -> false
        }
    }
val Annotation.unitSize: Int
    get() {
        return when (this) {
            is unit_8_t -> 1
            is unit_16_t -> 2
            is unit_32_t -> 4
            else -> throw IllegalStateException("annotation must be unit_8_t, unit_16_t or unit_32_t.")
        }
    }


private val sizeOfClasses = mutableMapOf<Class<*>, Int>()
fun sizeOf(klass: KClass<*>): Int {
    return sizeOf(klass.java)
}

fun sizeOf(clazz: Class<*>): Int {
    return sizeOfClasses.getOrPut(clazz) {
        clazz.declaredFields.sumOf { sizeOf(it) }
    }
}

private val sizeOfFields = mutableMapOf<Field, Int>()
fun sizeOf(field: Field): Int {
    return sizeOfFields.getOrPut(field) {
        field.declaredAnnotations
            .filter { it.isUnitAnnotation }
            .takeIf { it.size < 2 }
            ?.sumOf { if (it is unit_size_t_group) sizeOf(field.type.kotlin) else it.unitSize }
            ?: throw IllegalStateException(
                "Field ${field.name} can only have 1 annotation" +
                        " with unit_8_t, unit_16_t, unit_32_t or unit_size_t_group"
            )
    }
}