/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater

import java.io.File

/**
 * Created by chentao.joe on 2022/7/29
 * @author chentao.joe@bytedance.com
 */
internal interface CodeGenerator {

    val className: String

    val simpleClassName: String

    fun generateJavaSource(dir: File)
}


internal interface DependencyCodeGenerator : CodeGenerator {
    fun dependencies(): List<CodeGenerator>
}

internal fun CodeGenerator.allDependencies(): List<CodeGenerator>? {
    val dependencyList = mutableListOf<CodeGenerator>()
    // 1. check whether current CodeGenerator is DependencyCodeGenerator
    if (this is DependencyCodeGenerator) {
        // 2. add all dependencies
        val dependencies = dependencies()
        dependencyList.addAll(dependencies)
        // 3. loop all dependencies to find dependency's dependencies
        dependencies.filterIsInstance<DependencyCodeGenerator>()
            .flatMap { it.allDependencies() ?: emptyList() }
            .takeIf { it.isNotEmpty() }
            ?.let { dependencyList.addAll(it) }
    }
    return dependencyList.takeIf { it.isNotEmpty() }
}

internal interface IntermediateCodeGenerator : CodeGenerator {
    fun compose(others: List<IntermediateCodeGenerator>)
}


internal fun List<CodeGenerator>.finishIntermediate(): List<CodeGenerator> {
    // 1. filter IntermediateCodeGenerator and NonIntermediateCodeGenerator
    val intermediates = mutableListOf<IntermediateCodeGenerator>()
    val nonIntermediates = mutableListOf<CodeGenerator>()
    forEach { codeGenerator ->
        if (codeGenerator is IntermediateCodeGenerator) {
            intermediates.add(codeGenerator)
        } else {
            nonIntermediates.add(codeGenerator)
        }
    }
    // 2. collect IntermediateCodeGenerator by className
    val classToIntermediates = mutableMapOf<Class<*>, MutableList<IntermediateCodeGenerator>>()
    intermediates.forEach { intermediate ->
        classToIntermediates
            .getOrPut(intermediate::class.java) { mutableListOf() }
            .add(intermediate)
    }
    classToIntermediates.forEach { (_, list) ->
        list.first().compose(list)
    }
    // 3. get all first element of IntermediateCodeGenerator to refill list
    return nonIntermediates.plus(classToIntermediates.map { it.value.first() })
}