/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater

import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.*
import com.bytedance.ultimate.inflater.plugin.internal.resource.ResourceManager
import com.bytedance.ultimate.inflater.plugin.internal.toHex
import java.util.concurrent.atomic.AtomicBoolean

/**
 * Created by ChengTao(chentao.joe@bytedance.com) on 2022/9/16.
 */
internal class LayoutCreatorFactoryCodeGenerator(
    private val layoutSimpleName: String
) : AbsCodeGenerator(), DependencyCodeGenerator {
    override val templateName: String
        get() = TEMPLATE

    override val classPackage: String
        get() = PACKAGE

    override val simpleClassName: String
        get() = "${layoutSimpleName}_LayoutCreatorFactory"

    private val layoutId by lazy { ResourceManager.getLayoutId(layoutSimpleName) }

    private val layoutCreatorFactoryVariants = mutableMapOf<LayoutNameInfo, CodeGenerator>()

    private val dependenciesCalled = AtomicBoolean(false)
    override fun dependencies(): List<CodeGenerator> {
        if (dependenciesCalled.compareAndSet(false, true)) {
            val layoutInfoList = ResourceManager.getAllLayoutInfo(layoutSimpleName)
            if (layoutInfoList.isEmpty()) {
                throw IllegalStateException("Cannot find layout with name : $layoutSimpleName")
            }
            layoutInfoList.associateWith { VariantLayoutCreatorFactoryCodeGenerator(it) }.let {
                layoutCreatorFactoryVariants.putAll(it)
            }
        }
        return layoutCreatorFactoryVariants.values
            .plus(
                LayoutNameToIdRegistryImplCodeGenerator(
                    LayoutNameToId(
                        layoutSimpleName,
                        layoutId,
                        layoutId.toHex()
                    )
                )
            )
            .plus(
                LayoutCreatorFactoryRegistryImplCodeGenerator(
                    LayoutIdToFactory(layoutId, layoutId.toHex(), className)
                )
            )
    }

    override fun preGenerateJavaSource() {
        super.preGenerateJavaSource()
        if (dependenciesCalled.get().not()) {
            throw IllegalStateException("dependencies is not called")
        }
    }

    override fun createDataModel(): IGenerateModel {
        val variantList = layoutCreatorFactoryVariants.map { entry ->
            val layoutNameInfo = entry.key
            val codeGenerator = entry.value
            LayoutCreatorFactoryVariant(
                codeGenerator.className,
                codeGenerator.simpleClassName,
                layoutNameInfo.isLand,
                layoutNameInfo.isDefault,
                layoutNameInfo.version,
                layoutNameInfo.version != -1
            )
        }
        val landLayoutCreatorFactories = variantList.filter { it.isLand }
            .sortedByDescending { it.version }
        val versionLayoutCreatorFactories = variantList
            .filter { it.isLand.not() && it.isDefault.not() && it.version > 0 }
            .sortedByDescending { it.version }
        val defaultLayoutCreatorFactory = variantList.find { it.isDefault }
        val hasLand = landLayoutCreatorFactories.isNotEmpty()
        val hasMultipleVersion = variantList.find { it.version > 0 } != null
        val importClassList = variantList.map { it.className }.toMutableList().apply {
            if (hasLand) {
                addAll(landImportClasses)
            }
            if (hasMultipleVersion) {
                addAll(versionImportClasses)
            }
        }
        return LayoutCreatorFactoryModel(
            classPackage,
            className,
            simpleClassName,
            layoutSimpleName,
            importClassList,
            hasLand,
            hasMultipleVersion,
            landLayoutCreatorFactories,
            versionLayoutCreatorFactories,
            defaultLayoutCreatorFactory
        )
    }

    companion object {

        private const val PACKAGE = "com.bytedance.ultimate.inflater.internal.ui.layout"

        private const val TEMPLATE = "LayoutCreatorFactory"

        private val landImportClasses = listOf(
            "com.bytedance.ultimate.inflater.UltimateInflater",
            "android.content.res.Configuration"
        )

        private val versionImportClasses = listOf(
            "android.os.Build"
        )
    }
}