/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater

import com.bytedance.ultimate.inflater.plugin.arsc.single.SingleNodeResXmlTree
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.IGenerateModel
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.LayoutCreatorFactoryVariantModel
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.LayoutNameInfo
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.LayoutViewCreatorModel
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.RawViewCreatorModel
import com.bytedance.ultimate.inflater.plugin.internal.config.ConfigManager
import com.bytedance.ultimate.inflater.plugin.internal.hexToInt
import com.bytedance.ultimate.inflater.plugin.internal.resource.ResourceManager
import java.util.concurrent.atomic.AtomicBoolean

internal class VariantLayoutCreatorFactoryCodeGenerator(
    private val layoutNameInfo: LayoutNameInfo
) : AbsCodeGenerator(), DependencyCodeGenerator {

    override val classPackage: String
        get() = PACKAGE

    override val simpleClassName: String by lazy {
        "${layoutNameInfo.layoutSimpleName}_" +
                layoutNameInfo.layoutSimpleVariant.replace("-", "_") +
                "_LayoutCreatorFactory"
    }

    override val templateName: String
        get() = TEMPLATE

    private val layoutId by lazy { ResourceManager.getLayoutId(layoutNameInfo.layoutSimpleName) }

    private val viewCreators = mutableListOf<CodeGenerator>()

    private val includeLayouts = mutableListOf<CodeGenerator>()

    private val viewCreatorList = mutableListOf<LayoutViewCreatorModel>()

    private var isMergeLayout = false

    private val inflated = AtomicBoolean(false)

    override fun dependencies(): List<CodeGenerator> {
        if (inflated.get().not()) {
            synchronized(this) {
                if (inflated.compareAndSet(false, true)) {
                    inflate(ResourceManager.getSingleNodeResXmlTree(layoutNameInfo.layoutFullName))
                }
            }
        }
        return listOf(includeLayouts, viewCreators).flatten()
    }

    override fun preGenerateJavaSource() {
        super.preGenerateJavaSource()
        if (inflated.get().not()) {
            throw IllegalStateException("layout ${layoutNameInfo.layoutFullName} has not be inflated.")
        }
    }

    override fun createDataModel(): IGenerateModel {
        val importClassList = viewCreatorList
            .mapNotNull { it.viewCreatorClassName }.toSet().toList()
        return LayoutCreatorFactoryVariantModel(
            classPackage,
            className,
            simpleClassName,
            layoutId,
            isMergeLayout,
            importClassList,
            ConfigManager.appCompatViewInflaterClass,
            viewCreatorList
        )
    }

    private fun inflate(root: SingleNodeResXmlTree) {
        when (root.viewName) {
            "merge" -> {
                isMergeLayout = true
                rInflateChildren(root, 1, -1)
            }

            "include" -> {
                throw IllegalStateException(
                    "<include /> cannot be the root element in layout ${layoutNameInfo.layoutFullName}"
                )
            }

            else -> {
                rInflate(root, 0, -1)
            }
        }
    }

    private fun rInflate(node: SingleNodeResXmlTree, selfIndex: Int, parentIndex: Int): Int {
        var viewCreatorClassName: String? = null
        var viewCreatorSimpleClassName: String? = null
        var isInclude = false
        var includeLayoutId: Int? = null
        var includeLayoutName: String? = null
        var includeId: Int? = null
        var includeVisibility: Int? = null
        when (node.viewName) {
            "merge" -> {
                throw IllegalStateException(
                    "<merge /> must be the root element in layout ${layoutNameInfo.layoutFullName}"
                )
            }

            "include" -> {
                isInclude = true
                includeLayoutId = node.requireAttribute<String>("layout").hexToInt()
                includeLayoutName = ResourceManager.getLayoutName(includeLayoutId)
                includeId = node.getAttribute<Any>("id")?.toString()?.hexToInt()
                includeVisibility = node.getAttribute<Any>("visibility")?.toString()?.toInt()
                includeLayouts.add(
                    LayoutCreatorFactoryCodeGenerator(
                        ResourceManager.getLayoutName(includeLayoutId)
                    )
                )
            }

            else -> {
                val viewCreatorCodeGenerator = ViewCreatorCodeGenerator(RawViewCreatorModel(node))
                viewCreators.add(viewCreatorCodeGenerator)
                viewCreatorClassName = viewCreatorCodeGenerator.className
                viewCreatorSimpleClassName = viewCreatorCodeGenerator.simpleClassName
            }
        }
        val attributeDataProviderSimpleClassName = generateAttributeDataProviderSimpleClassName(
            node, selfIndex
        )
        val attributeDataString = node.data.joinToString { it.toString() }
        var selfVal = "val$selfIndex"
        if (selfIndex == 0) {
            selfVal = "root"
        }
        var parentVal: String? = "val${parentIndex}"
        if (parentIndex == -1) {
            parentVal = if (isMergeLayout) "root" else null
        }
        if (parentIndex == 0) {
            parentVal = "root"
        }
        viewCreatorList.add(
            LayoutViewCreatorModel(
                viewCreatorClassName,
                viewCreatorSimpleClassName,
                attributeDataProviderSimpleClassName,
                attributeDataString,
                selfVal,
                parentVal,
                node.getAttribute<String>("theme")?.hexToInt(),
                isInclude,
                includeLayoutId,
                includeLayoutName,
                includeId,
                includeVisibility
            )
        )
        return rInflateChildren(node, selfIndex + 1, selfIndex)
    }

    private fun rInflateChildren(
        parent: SingleNodeResXmlTree,
        childStartIndex: Int,
        parentIndex: Int
    ): Int {
        var childIndex = childStartIndex
        parent.children?.forEach { child ->
            childIndex = rInflate(child, childIndex, parentIndex)
        }
        return childIndex
    }

    private fun generateAttributeDataProviderSimpleClassName(
        node: SingleNodeResXmlTree,
        selfIndex: Int
    ): String {
        val suffix = when (node.viewName) {
            "view" -> {
                buildString {
                    append("_")
                    append(
                        node.requireAttribute<String>("class").replace(".", "_")
                    )
                }
            }

            "include" -> {
                buildString {
                    append("_")
                    append(
                        ResourceManager.getLayoutName(
                            node.requireAttribute<String>("layout").hexToInt()
                        )
                    )
                }
            }

            else -> ""
        }
        val viewDescription = node.viewName.replace(".", "_")
        return "${viewDescription}${suffix}_${selfIndex}_AttributeDataProvider"
    }

    companion object {
        private const val PACKAGE = "com.bytedance.ultimate.inflater.internal.ui.layout"
        private const val TEMPLATE = "VariantLayoutCreatorFactory"
    }
}