/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater

import com.bytedance.ultimate.inflater.plugin.IUltimateInflaterExtension.Companion.APP_COMPAT_VIEW_INFLATER_ANDROIDX
import com.bytedance.ultimate.inflater.plugin.IUltimateInflaterExtension.Companion.APP_COMPAT_VIEW_INFLATER_ANDROIDX_MATERIAL
import com.bytedance.ultimate.inflater.plugin.IUltimateInflaterExtension.Companion.APP_COMPAT_VIEW_INFLATER_SUPPORT
import com.bytedance.ultimate.inflater.plugin.IUltimateInflaterExtension.Companion.APP_COMPAT_VIEW_INFLATER_SUPPORT_MATERIAL
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.AppCompatViewCreatorModel
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.IGenerateModel
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.NormalViewCreatorModel
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.RawViewCreatorModel
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.ViewCreationCondition
import com.bytedance.ultimate.inflater.plugin.internal.config.ConfigManager
import com.bytedance.ultimate.inflater.plugin.internal.config.LayoutInflaterFactoryConfig

/**
 * Created by chentao.joe on 2022/7/29
 * @author chentao.joe@bytedance.com
 */
internal class ViewCreatorCodeGenerator(rawDatModel: RawViewCreatorModel) : AbsCodeGenerator() {

    private val viewName: String

    private val _simpleClassName: String

    private var isAppCompatView = false

    override val classPackage: String
        get() = PACKAGE

    override val templateName: String
        get() = if (isAppCompatView) APP_COMPAT_TEMPLATE else NORMAL_TEMPLATE

    override val simpleClassName: String
        get() = _simpleClassName

    private val appCompatViewInflateClass by lazy {
        ConfigManager.appCompatViewInflaterClass
    }

    private var _viewClassName: String? = null

    private val viewClassName: String
        get() = _viewClassName
            ?: throw IllegalStateException("Cannot get view class name.")

    init {
        var viewName = rawDatModel.node.viewName
        if (viewName == "view") {
            viewName = rawDatModel.node.requireAttribute("class")
        }
        this.viewName = viewName
        _simpleClassName = "${viewName.replace(".", "_")}_ViewCreator"
        if (viewName.contains(".").not()) {
            isAppCompatView = supportAppCompatViews.contains(viewName)
            if (isAppCompatView.not()) {
                if (appCompatViewInflateClass == APP_COMPAT_VIEW_INFLATER_ANDROIDX
                    || appCompatViewInflateClass == APP_COMPAT_VIEW_INFLATER_ANDROIDX_MATERIAL
                ) {
                    isAppCompatView = androidxAppCompatViews.contains(viewName)
                }
            }
        }
    }

    override fun createDataModel(): IGenerateModel {
        val viewClassName = if (viewName.contains(".")) {
            viewName
        } else {
            getViewClassNameByViewName(viewName)
        }
        _viewClassName = viewClassName
        return if (isAppCompatView) {
            return createAppCompatViewCreatorModel()
        } else {
            val isRequireMainThread = ConfigManager.isViewRequireMainThread(viewClassName)
            val isRequireActivityContext = ConfigManager.isViewRequireActivityContext(viewClassName)
            val hasViewCreationCondition = isRequireMainThread || isRequireActivityContext
            val layoutInflaterFactory = ConfigManager.getLayoutInflaterFactory(viewClassName)
            val isDefaultLayoutInflaterFactory =
                layoutInflaterFactory == LayoutInflaterFactoryConfig.DEFAULT_FACTORY
            NormalViewCreatorModel(
                classPackage,
                className,
                simpleClassName,
                viewClassName,
                hasViewCreationCondition,
                isRequireMainThread,
                isRequireActivityContext,
                isDefaultLayoutInflaterFactory,
                layoutInflaterFactory
            )
        }
    }

    private fun createAppCompatViewCreatorModel(): AppCompatViewCreatorModel {
        val conditionToViewClassName = mutableMapOf<String, String>()
        var hasMaterialAppCompatView = false
        // widget
        val widgetViewClassName = viewClassName
        conditionToViewClassName["Widget"] = widgetViewClassName
        // androidx
        if (appCompatViewInflateClass == APP_COMPAT_VIEW_INFLATER_ANDROIDX ||
            appCompatViewInflateClass == APP_COMPAT_VIEW_INFLATER_ANDROIDX_MATERIAL
        ) {
            conditionToViewClassName["AndroidXAppCompat"] =
                "androidx.appcompat.widget.AppCompat$viewName"
        }
        // support
        if (appCompatViewInflateClass == APP_COMPAT_VIEW_INFLATER_SUPPORT ||
            appCompatViewInflateClass == APP_COMPAT_VIEW_INFLATER_SUPPORT_MATERIAL
        ) {
            conditionToViewClassName["SupportAppCompat"] =
                "android.support.v7.widget.AppCompat$viewName"
        }
        // androidx-material, support-material
        if (appCompatViewInflateClass == APP_COMPAT_VIEW_INFLATER_ANDROIDX_MATERIAL ||
            appCompatViewInflateClass == APP_COMPAT_VIEW_INFLATER_SUPPORT_MATERIAL
        ) {
            materialAppCompatViews[viewName]?.let { materialViewClassName ->
                conditionToViewClassName["MaterialAppCompat"] = materialViewClassName
                hasMaterialAppCompatView = true
            }
        }
        val viewCreationConditionList = conditionToViewClassName.map { (condition, viewClassName) ->
            val isRequireMainThread = ConfigManager.isViewRequireMainThread(viewClassName)
            val isRequireActivityContext = ConfigManager.isViewRequireActivityContext(viewClassName)
            val hasCondition = isRequireMainThread || isRequireActivityContext
            val layoutInflaterFactory = ConfigManager.getLayoutInflaterFactory(viewClassName)
            val isDefaultLayoutInflaterFactory =
                layoutInflaterFactory == LayoutInflaterFactoryConfig.DEFAULT_FACTORY
            ViewCreationCondition(
                condition,
                viewName,
                viewClassName,
                hasCondition,
                isRequireMainThread,
                isRequireActivityContext,
                isDefaultLayoutInflaterFactory,
                layoutInflaterFactory
            )
        }.toMutableList()
        // add automatic condition
        viewCreationConditionList.add(
            ViewCreationCondition(
                "Automatic",
                viewName,
                viewName,
                hasCondition = true,
                isRequireMainTread = ConfigManager.isViewRequireMainThread(viewName),
                isRequireActivityContext = true,
                isDefaultLayoutInflaterFactory = true,
                layoutInflaterFactoryClassName = LayoutInflaterFactoryConfig.DEFAULT_FACTORY
            )
        )
        return AppCompatViewCreatorModel(
            classPackage,
            className,
            simpleClassName,
            hasMaterialAppCompatView,
            viewCreationConditionList
        )
    }

    private fun getViewClassNameByViewName(viewName: String): String {
        return when (viewName) {
            "fragment",
            "blank" -> throw IllegalStateException("cannot create view name with : $viewName")

            "View",
            "ViewStub",
            "SurfaceView",
            "TextureView" -> "android.view.${viewName}"

            "MediaRouteButton",
            "FragmentBreadCrumbs" -> "android.app.${viewName}"

            "WebView" -> "android.webkit.${viewName}"
            else -> "android.widget.${viewName}"
        }
    }

    companion object {
        private const val PACKAGE = "com.bytedance.ultimate.inflater.internal.ui.view"
        private const val NORMAL_TEMPLATE = "NormalViewCreator"
        private const val APP_COMPAT_TEMPLATE = "AppCompatViewCreator"

        private val supportAppCompatViews = setOf(
            "TextView",
            "ImageView",
            "Button",
            "EditText",
            "Spinner",
            "ImageButton",
            "CheckBox",
            "RadioButton",
            "CheckedTextView",
            "AutoCompleteTextView",
            "MultiAutoCompleteTextView",
            "RatingBar",
            "SeekBar"
        )

        private val androidxAppCompatViews = supportAppCompatViews
            .plus("ToggleButton")

        private val materialAppCompatViews = mapOf(
            "Button" to "com.google.android.material.button.MaterialButton",
            "CheckBox" to "com.google.android.material.checkbox.MaterialCheckBox",
            "RadioButton" to "com.google.android.material.radiobutton.MaterialRadioButton",
            "TextView" to "com.google.android.material.textview.MaterialTextView",
            "AutoCompleteTextView" to "com.google.android.material.textfield.MaterialAutoCompleteTextView"
        )
    }
}
