/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model

import java.util.regex.Pattern

/**
 * Created by chentao.joe on 2022/9/18
 * @author chentao.joe@bytedance.com
 */
internal data class LayoutNameInfo(
    /**
     * res/layout/layout1.xml
     * res/layout-v21/layout_v21.xml
     * res/layout-land/layout_land.xml
     * res/layout-land-v21/layout_land_v21.xml
     * ...
     */
    val layoutFullName: String,
    /**
     * layout1
     * layout_v21
     * layout_land
     * layout_land_v21
     * ...
     */
    val layoutSimpleName: String,
    /**
     * layout, layout-v21, layout-land, layout-land-v21 and so on
     */
    val layoutVariant: String,
    /**
     * default, v21, land , land-v21 and so on
     */
    val layoutSimpleVariant: String,
    /**
     * indicate the layout [layoutFullName] is land or portrait.
     */
    val isLand: Boolean,
    /**
     * indicate the layout [layoutFullName] is default or not.
     */
    val isDefault: Boolean,
    /**
     * indicate layout version, -1 indicate no version.
     */
    val version: Int
) {

    companion object {

        private val LAYOUT_PATTEN = Pattern.compile(
            "res/layout(-land)?(-v[0-9]{1,2})?/[1-9a-z_]*.xml"
        )

        fun parse(layoutFullName: String): LayoutNameInfo {
            if (LAYOUT_PATTEN.matcher(layoutFullName).find().not()) {
                throw IllegalStateException("Invalid layout : $layoutFullName")
            }
            val splitList = layoutFullName.split("/")
            val layoutSimpleName = splitList[2].removeSuffix(".xml")
            val layoutVariant = splitList[1]
            val layoutSimpleVariant = layoutVariant.indexOf("-").takeIf { it != -1 }
                ?.let { layoutVariant.substring(it + 1) } ?: "default"
            val isLand = layoutSimpleVariant.startsWith("land")
            val isDefault = isLand.not() && layoutSimpleVariant == "default"
            val version = layoutVariant.indexOf("-v").takeIf { it != -1 }
                ?.let { layoutVariant.substring(it + 2).toInt() } ?: -1
            return LayoutNameInfo(
                layoutFullName,
                layoutSimpleName,
                layoutVariant,
                layoutSimpleVariant,
                isLand,
                isDefault,
                version
            )
        }
    }
}