/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.config

import com.bytedance.ultimate.inflater.plugin.IUltimateInflaterExtension
import com.bytedance.ultimate.inflater.plugin.api.compat.ConfigFileCollection
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.LayoutNameInfo
import com.bytedance.ultimate.inflater.plugin.internal.isZipFile
import com.bytedance.ultimate.inflater.plugin.internal.toSystemPath
import com.moandjiezana.toml.Toml
import org.gradle.api.Project
import org.gradle.api.file.FileCollection
import java.io.File
import java.util.zip.ZipFile

/**
 * Created by chentao.joe on 2022/9/25
 * @author chentao.joe@bytedance.com
 */
internal object ConfigManager {

    // region : extension config

    private lateinit var extensionConfig: ExtensionConfig

    val appCompatViewInflaterClass: String
        get() = extensionConfig.appCompatViewInflateClass

    fun parseExtensionConfig(project: Project) {
        val extension = project.extensions.getByName(IUltimateInflaterExtension.NAME)
        if (extension !is IUltimateInflaterExtension) {
            throw IllegalStateException(
                "Extension with name \"${IUltimateInflaterExtension.NAME}\" "
                        + "must be type of ${IUltimateInflaterExtension::class.java}"
            )
        }
        // app compat inflate class
        val appCompatInflateClass = cheAppCompatInflateClass(extension.appCompatViewInflaterClass())
        extensionConfig = ExtensionConfig(appCompatInflateClass)
    }

    private fun cheAppCompatInflateClass(clazz: String): String {
        return when (clazz) {
            IUltimateInflaterExtension.APP_COMPAT_VIEW_INFLATER_AUTOMATIC,
            IUltimateInflaterExtension.APP_COMPAT_VIEW_INFLATER_WIDGET,
            IUltimateInflaterExtension.APP_COMPAT_VIEW_INFLATER_ANDROIDX,
            IUltimateInflaterExtension.APP_COMPAT_VIEW_INFLATER_SUPPORT,
            IUltimateInflaterExtension.APP_COMPAT_VIEW_INFLATER_ANDROIDX_MATERIAL,
            IUltimateInflaterExtension.APP_COMPAT_VIEW_INFLATER_SUPPORT_MATERIAL -> clazz

            else -> if (clazz.contains(".")) {
                throw IllegalStateException(
                    "appCompatInflateClass $clazz must be automatic, widget, androidx, support," +
                            "android-material, support material or a full path of class."
                )
            } else {
                clazz
            }
        }
    }

    // endregion

    // region: code generation config

    private const val LAYOUT_CONFIG = "layout-config.toml"
    private const val VIEW_CONFIG = "view-config.toml"
    private const val LAYOUT_INFLATER_FACTORY_CONFIG = "layout-inflater-factory-config.toml"

    private const val META_INF_DIR = "META-INF/ultimate-inflater"
    private const val PROJECT_CONFIG_FILE_DIR = "src/main/resources/${META_INF_DIR}"

    private const val PROJECT_LAYOUT_CONFIG_FILE = "$PROJECT_CONFIG_FILE_DIR/$LAYOUT_CONFIG"
    private const val PROJECT_VIEW_CONFIG_FILE = "$PROJECT_CONFIG_FILE_DIR/$VIEW_CONFIG"
    private const val PROJECT_LAYOUT_INFLATER_FACTORY_CONFIG_FILE =
        "$PROJECT_CONFIG_FILE_DIR/$LAYOUT_INFLATER_FACTORY_CONFIG"

    private const val META_INF_LAYOUT_CONFIG_FILE = "$META_INF_DIR/$LAYOUT_CONFIG"
    private const val META_INF_VIEW_CONFIG_FILE = "$META_INF_DIR/$VIEW_CONFIG"
    private const val META_INF_LAYOUT_INFLATER_FACTORY_CONFIG_FILE =
        "$META_INF_DIR/$LAYOUT_INFLATER_FACTORY_CONFIG"

    private lateinit var layoutConfigs: Map<String, LayoutConfig>
    val layoutVariantFilter: ((LayoutNameInfo) -> Boolean)
        get() = { layoutNameInfo ->
            layoutConfigs[layoutNameInfo.layoutSimpleName]?.supportLayouts
                ?.contains(layoutNameInfo.layoutFullName) ?:
            // only support default layout
            (layoutNameInfo.layoutFullName == "res/layout/${layoutNameInfo.layoutSimpleName}.xml")
        }
    val enableLayouts: Set<String>
        get() = layoutConfigs.keys

    private lateinit var viewConfigs: Map<String, ViewConfig>

    private lateinit var layoutInflaterFactoryConfigs: Map<String, LayoutInflaterFactoryConfig>

    val allLayoutInflaterFactories: Set<String>
        get() = layoutInflaterFactoryConfigs.map { it.value.factoryClassName }
            .filter { it != LayoutInflaterFactoryConfig.DEFAULT_FACTORY }
            .toSet()

    fun isViewRequireMainThread(viewClassName: String): Boolean {
        return viewConfigs[viewClassName]?.isRequireMainThread ?: false
    }

    fun isViewRequireActivityContext(viewClassName: String): Boolean {
        val viewConfig = viewConfigs[viewClassName]
        if (viewConfig != null && viewConfig.isRequireActivityContext) {
            return true
        }
        val layoutInflaterFactoryConfig = layoutInflaterFactoryConfigs[viewClassName]
        if (layoutInflaterFactoryConfig?.factoryClassName == LayoutInflaterFactoryConfig.DEFAULT_FACTORY) {
            return true
        }
        return false
    }

    fun getLayoutInflaterFactory(viewClassName: String): String? {
        return layoutInflaterFactoryConfigs[viewClassName]?.factoryClassName
    }


    private class ConfigFileInfo {
        private val _layoutConfigFiles = mutableMapOf<String, LayoutConfig>()
        val layoutConfigFiles: Map<String, LayoutConfig>
            get() = _layoutConfigFiles

        fun addLayoutConfig(config: LayoutConfig) {
            _layoutConfigFiles[config.name] = config
        }

        private val _viewConfigFiles = mutableMapOf<String, ViewConfig>()

        val viewConfigFiles: Map<String, ViewConfig>
            get() = _viewConfigFiles

        fun addViewConfigFile(config: ViewConfig) {
            _viewConfigFiles[config.className] = config
        }

        private val _layoutInflaterFactoryConfigFiles =
            mutableMapOf<String, LayoutInflaterFactoryConfig>()

        val layoutInflaterFactoryConfigFiles: Map<String, LayoutInflaterFactoryConfig>
            get() = _layoutInflaterFactoryConfigFiles

        fun addLayoutInflaterConfigFile(config: LayoutInflaterFactoryConfig) {
            _layoutInflaterFactoryConfigFiles[config.viewClassName] = config
        }
    }

    private interface ConfigList

    private class LayoutConfigList(val list: List<LayoutConfig>) : ConfigList

    private class ViewConfigList(val list: List<ViewConfig>) : ConfigList

    private class LayoutInflaterFactoryConfigList(
        val list: List<LayoutInflaterFactoryConfig>
    ) : ConfigList

    /**
     * parse config files
     *
     * @param configFiles a [ConfigFileCollection]
     */
    fun parseConfigFiles(configFiles: ConfigFileCollection) {
        val appProjectConfigFiles = configFiles.appProjectConfigFiles.files
        val subProjectConfigFiles = collectSubProjectConfigFiles(configFiles.subProjectConfigFiles)
        val externalLibConfigFiles = configFiles.externalLibConfigFiles.files
        // priority of config file :
        // appProjectConfigFiles > subProjectConfigFiles > externalLibConfigFiles
        val appProjectConfigFileInfo = parseConfigFiles(appProjectConfigFiles)
        val subProjectConfigFileInfo = parseConfigFiles(subProjectConfigFiles)
        val externalLibConfigFileInfo = parseConfigFiles(externalLibConfigFiles)
        // layout-config
        val layoutConfigs = mutableMapOf<String, LayoutConfig>()
        layoutConfigs.putAll(externalLibConfigFileInfo.layoutConfigFiles)
        layoutConfigs.putAll(subProjectConfigFileInfo.layoutConfigFiles)
        layoutConfigs.putAll(appProjectConfigFileInfo.layoutConfigFiles)
        this.layoutConfigs = layoutConfigs.toMap()
        // view-config
        val viewConfigs = mutableMapOf<String, ViewConfig>()
        viewConfigs.putAll(externalLibConfigFileInfo.viewConfigFiles)
        viewConfigs.putAll(subProjectConfigFileInfo.viewConfigFiles)
        viewConfigs.putAll(appProjectConfigFileInfo.viewConfigFiles)
        this.viewConfigs = viewConfigs.toMap()
        // layout-inflater-factory-config
        val layoutInflaterFactoryConfigs = mutableMapOf<String, LayoutInflaterFactoryConfig>()
        layoutInflaterFactoryConfigs.putAll(externalLibConfigFileInfo.layoutInflaterFactoryConfigFiles)
        layoutInflaterFactoryConfigs.putAll(subProjectConfigFileInfo.layoutInflaterFactoryConfigFiles)
        layoutInflaterFactoryConfigs.putAll(appProjectConfigFileInfo.layoutInflaterFactoryConfigFiles)
        this.layoutInflaterFactoryConfigs = layoutInflaterFactoryConfigs
    }

    private fun collectSubProjectConfigFiles(subProjectConfigFiles: FileCollection): List<File> {
        val buildPath = "/build/".toSystemPath()
        return subProjectConfigFiles.files.mapNotNull { file ->
            val absolutePath = file.absolutePath
            val buildIndex = absolutePath.lastIndexOf(buildPath)
            if (buildIndex == -1) {
                return@mapNotNull null
            }
            val projectPath = absolutePath.substring(0, buildIndex)
            return@mapNotNull listOf(
                File(projectPath, PROJECT_LAYOUT_CONFIG_FILE),
                File(projectPath, PROJECT_VIEW_CONFIG_FILE),
                File(projectPath, PROJECT_LAYOUT_INFLATER_FACTORY_CONFIG_FILE)
            ).filter { it.exists() }
        }.flatten()
    }

    private fun parseConfigFiles(files: Collection<File>): ConfigFileInfo {
        val result = ConfigFileInfo()
        files.mapNotNull { file ->
            when {
                file.isZipFile -> parseZipConfigFile(file)
                else -> parseNormalConfigFile(file)?.let { listOf(it) }
            }
        }.flatten().forEach { configList ->
            when (configList) {
                is LayoutConfigList -> {
                    configList.list.forEach { config ->
                        result.addLayoutConfig(config)
                    }
                }

                is ViewConfigList -> {
                    configList.list.forEach { config ->
                        result.addViewConfigFile(config)
                    }
                }

                is LayoutInflaterFactoryConfigList -> {
                    configList.list.forEach { config ->
                        result.addLayoutInflaterConfigFile(config)
                    }
                }
            }
        }
        return result
    }

    private fun parseZipConfigFile(file: File): List<ConfigList>? {
        return ZipFile(file).use { zipFile ->
            zipFile.entries().asSequence().mapNotNull { zipEntry ->
                val entryBytes by lazy {
                    zipFile.getInputStream(zipEntry).use { it.readBytes() }
                }
                when (zipEntry.name) {
                    META_INF_LAYOUT_CONFIG_FILE -> parseLayoutConfig(entryBytes)
                    META_INF_VIEW_CONFIG_FILE -> parseViewConfig(entryBytes)
                    META_INF_LAYOUT_INFLATER_FACTORY_CONFIG_FILE -> parseLayoutInflaterFactoryConfig(
                        entryBytes
                    )

                    else -> null
                }
            }.toList().takeIf { it.isNotEmpty() }
        }
    }

    private fun parseNormalConfigFile(file: File): ConfigList? {
        val fileBytes by lazy { file.readBytes() }
        return when (file.name) {
            LAYOUT_CONFIG -> parseLayoutConfig(fileBytes)
            VIEW_CONFIG -> parseViewConfig(fileBytes)
            LAYOUT_INFLATER_FACTORY_CONFIG -> parseLayoutInflaterFactoryConfig(fileBytes)
            else -> null
        }
    }

    @Suppress("UNCHECKED_CAST")
    private fun parseLayoutConfig(tomlBytes: ByteArray): LayoutConfigList {
        return Toml().read(String(tomlBytes)).toMap()
            .map { (layoutName, layoutConfig) ->
                val ignoreDefaultVariant = runCatching {
                    (layoutConfig as Map<String, Any>)["ignoreDefaultVariant"] as Boolean
                }.getOrElse { false }
                val variants = runCatching {
                    (layoutConfig as Map<String, Any>)["variants"] as List<String>
                }.getOrElse { emptyList() }.toMutableSet().also { set ->
                    if (ignoreDefaultVariant.not()) {
                        set.add("")
                    }
                }.toSet()
                LayoutConfig(layoutName, variants)
            }
            .let { LayoutConfigList(it) }
    }

    private fun parseViewConfig(tomlBytes: ByteArray): ViewConfigList {
        // 1. read all view config
        val toml = Toml().read(String(tomlBytes))
        val mainThreadViews = toml.getList<String>("main_thread")
            ?.toSet() ?: emptySet()
        val activityContextViews = toml.getList<String>("activity_context")
            ?.toSet() ?: emptySet()
        // 2. convert
        return mutableSetOf<String>().apply {
            addAll(mainThreadViews)
            addAll(activityContextViews)
        }.map { viewClassName ->
            ViewConfig(
                viewClassName,
                mainThreadViews.contains(viewClassName),
                activityContextViews.contains(viewClassName)
            )
        }.let { ViewConfigList(it) }
    }

    @Suppress("UNCHECKED_CAST")
    private fun parseLayoutInflaterFactoryConfig(tomlBytes: ByteArray): LayoutInflaterFactoryConfigList {
        val configMap = mutableMapOf<String, LayoutInflaterFactoryConfig>()
        Toml().read(String(tomlBytes)).toMap().forEach { (factoryClassName, viewClassNameList) ->
            viewClassNameList as List<String>
            viewClassNameList.forEach { viewClassName ->
                val oldFactory = configMap[viewClassName]
                if (oldFactory != null) {
                    throw IllegalStateException(
                        "View named $viewClassName config different LayoutInflater.Factory " +
                                "in one layout-inflater-factory config file."
                    )
                }
                configMap[viewClassName] = LayoutInflaterFactoryConfig(
                    viewClassName, factoryClassName.replace("\"", "")
                )
            }
        }
        return LayoutInflaterFactoryConfigList(configMap.map { it.value })
    }
    // endregion
}