/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal

import org.objectweb.asm.tree.AbstractInsnNode
import org.objectweb.asm.tree.InsnList
import java.io.File

internal val CPU_COUNT: Int
    get() {
        val cpuCount = runCatching {
            System.getProperty("cpu_count").toInt()
        }.getOrElse {
            Runtime.getRuntime().availableProcessors()
        }
        return if (cpuCount < 1) 1 else cpuCount
    }

internal fun String.capitalizeUS(): String {
    val string = this
    return buildString {
        string.forEachIndexed { index, char ->
            if (index == 0 && Character.isLowerCase(char)) {
                append(Character.toTitleCase(char))
            } else {
                append(char)
            }
        }
    }
}

internal fun String.toSystemPath() = this.replace("/", File.separator)

internal fun String.simpleClassName() = this.substring(this.lastIndexOf(".") + 1)

internal fun String.hexToInt(): Int {
    if (this.startsWith("0x") || this.startsWith("0X")) {
        return this.substring(2).hexToInt()
    }
    return toInt(16)
}

internal fun Int.toHex(): String {
    return "0x${Integer.toHexString(this)}"
}

internal fun <E, R> List<E>.findFirstNotNullExecutedValue(
    notFoundMessage: String,
    invoke: (e: E) -> R?
): R {
    forEach { item ->
        val value = invoke(item)
        if (value != null) {
            return value
        }
    }
    throw IllegalStateException(notFoundMessage)
}

internal val File.isZipFile
    get() = name.endsWith(".jar") || name.endsWith(".aar")

internal fun <T> measureTime(description: String, block: () -> T): T {
    val startTime = System.currentTimeMillis()
    return block().also {
        val timeCost = System.currentTimeMillis() - startTime
        println("$description use : ${timeCost}ms")
    }
}

internal fun <K, V> MutableMap<K, V>.removeAll(collection: Collection<K>) {
    collection.forEach { key ->
        remove(key)
    }
}

internal fun InsnList.asIterable(): Iterable<AbstractInsnNode> = Iterable { iterator() }
internal fun InsnList.find(opcode: Int) = asIterable().find { it.opcode == opcode }
internal fun InsnList.filter(predicate: (AbstractInsnNode) -> Boolean) =
    asIterable().filter(predicate)