/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.resource

import com.bytedance.ultimate.inflater.plugin.arsc.ResTable
import com.bytedance.ultimate.inflater.plugin.arsc.single.SingleNodeResXmlTree
import com.bytedance.ultimate.inflater.plugin.arsc.toResXmlTree
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.LayoutNameInfo
import com.bytedance.ultimate.inflater.plugin.internal.config.ConfigManager
import com.bytedance.ultimate.inflater.plugin.internal.hexToInt
import java.io.File
import java.util.concurrent.ConcurrentHashMap
import java.util.zip.ZipFile

/**
 * Created by ChengTao(chentao.joe@bytedance.com) on 2022/9/13.
 */
@Suppress("ObjectPropertyName")
internal object ResourceManager {

    private var _resTable: ResTable? = null

    private val resTable: ResTable
        get() = requireNotNull(_resTable) { "ResTable is null" }

    private var _applicationThemeId: Int = 0

    val applicationThemeId: Int
        get() = _applicationThemeId

    private val layoutFullNameToData = mutableMapOf<String, ByteArray>()

    private val layoutFullNameToSingleNodeResXmlTree =
        ConcurrentHashMap<String, SingleNodeResXmlTree>()

    fun parseResources(processResourcesDir: File) {
        val resourcesApk = processResourcesDir
            .listFiles { f -> f.isFile && f.name.endsWith(".ap_") }?.firstOrNull()
            ?: throw IllegalStateException(
                "Cannot find the .ap_ file in ${processResourcesDir.absoluteFile}"
            )
        ZipFile(resourcesApk).use { zipFile ->
            val entries = zipFile.entries()
            while (entries.hasMoreElements()) {
                val zipEntry = entries.nextElement()
                when {
                    zipEntry.name == "resources.arsc" -> {
                        _resTable = ResTable(
                            zipFile.getInputStream(zipEntry).use { it.readBytes() }
                        )
                    }

                    zipEntry.name.startsWith("res/layout") -> {
                        layoutFullNameToData[zipEntry.name] = zipFile.getInputStream(zipEntry).use {
                            it.readBytes()
                        }
                    }

                    zipEntry.name == "AndroidManifest.xml" -> {
                        zipFile.getInputStream(zipEntry).use { it.readBytes() }
                            .toResXmlTree()
                            .rootNode.children?.find { it.name == "application" }
                            ?.attributes?.get("theme")?.let { themeId ->
                                if (themeId is String) {
                                    _applicationThemeId = themeId.hexToInt()
                                }
                            }
                    }

                    else -> {
                        // do nothing
                    }
                }
            }
        }
    }

    fun getLayoutName(layoutId: Int): String {
        return resTable.getLayoutName(layoutId)
    }

    fun getLayoutId(layoutName: String): Int {
        return resTable.getLayoutId(layoutName)
    }

    fun getAllLayoutInfo(layoutSimpleName: String): List<LayoutNameInfo> {
        return layoutFullNameToData.keys
            .filter { it.endsWith("/$layoutSimpleName.xml") }
            .map { LayoutNameInfo.parse(it) }
            .filter(ConfigManager.layoutVariantFilter)
    }

    fun getSingleNodeResXmlTree(fullLayoutName: String): SingleNodeResXmlTree {
        val tree = layoutFullNameToSingleNodeResXmlTree[fullLayoutName]
        if (tree != null) {
            return tree
        }
        val layoutData = layoutFullNameToData[fullLayoutName]
        if (layoutData != null) {
            return layoutData.toResXmlTree()
                .asSingleNodeResXmlTree(fullLayoutName)
                .also { layoutFullNameToSingleNodeResXmlTree[fullLayoutName] to it }
        }
        throw IllegalStateException("Cannot find layout with name : $fullLayoutName")
    }

    fun release() {
        _resTable = null
        layoutFullNameToData.clear()
        layoutFullNameToSingleNodeResXmlTree.clear()
    }
}