/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.task

import com.bytedance.ultimate.inflater.plugin.api.compat.ConfigFileCollection
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.LayoutCreatorFactoryCodeGenerator
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.LayoutCreatorFactoryRegistryImplCodeGenerator
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.LayoutNameToIdRegistryImplCodeGenerator
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.UltimateInflaterApplicationThemeImplCodeGenerator
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.allDependencies
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.finishIntermediate
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.LayoutIdToFactory
import com.bytedance.ultimate.inflater.plugin.internal.codegen.inflater.model.LayoutNameToId
import com.bytedance.ultimate.inflater.plugin.internal.config.ConfigManager
import com.bytedance.ultimate.inflater.plugin.internal.resource.ResourceManager
import org.gradle.api.DefaultTask
import org.gradle.api.tasks.InputDirectory
import org.gradle.api.tasks.Nested
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.TaskAction
import java.io.File

/**
 * @author Tao Cheng (tao@paradisehell.org)
 */
internal open class GenerateSourcesTask : DefaultTask() {

    @InputDirectory
    lateinit var processResourcesDir: File

    @OutputDirectory
    lateinit var generateJavaSourceRoot: File

    @Nested
    lateinit var configFiles: ConfigFileCollection

    @TaskAction
    fun generate() {
        generateJavaSourceRoot.deleteRecursively()
        // 1. parse layout config file
        runCatching {
            ConfigManager.parseConfigFiles(configFiles)
            val layouts = ConfigManager.enableLayouts
            // 2. parse resources
            ResourceManager.parseResources(processResourcesDir)
            // 3. generate sources code and unreachable view jar
            generateSources(layouts)
            // 4. release ResourceManager
            ResourceManager.release()
        }.onFailure {
            it.printStackTrace()
        }.getOrThrow()
    }

    private fun generateSources(layouts: Set<String>) {
        // LayoutCreatorFactoryRegistryImplCodeGenerator and LayoutNameToIdRegistryImplCodeGenerator
        // must be called to generate source file.
        return listOf(LayoutCreatorFactoryRegistryImplCodeGenerator(LayoutIdToFactory.NONE))
            .plus(LayoutNameToIdRegistryImplCodeGenerator(LayoutNameToId.NONE))
            .plus(UltimateInflaterApplicationThemeImplCodeGenerator(ResourceManager.applicationThemeId))
            // add all LayoutCreatorFactoryCodeGenerator
            .plus(layouts.map { LayoutCreatorFactoryCodeGenerator(it) })
            // get all dependencies
            .flatMap { listOfNotNull(it.allDependencies()).flatten().plus(it) }
            // finish some Intermediate CodeGenerator like LayoutCreatorFactoryRegistryImplCodeGenerator
            // and LayoutNameToIdRegistryImplCodeGenerator, this kind of CodeGenerator can only
            // generate 1 source file.
            .finishIntermediate().associateBy { it.className }
            // real generate source file
            .forEach { (_, codeGenerator) ->
                codeGenerator.generateJavaSource(generateJavaSourceRoot)
            }
    }
}