/*
 * Copyright (C) 2023 ByteDance Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bytedance.ultimate.inflater.plugin.internal.transformer

import org.objectweb.asm.Opcodes
import org.objectweb.asm.tree.ClassNode
import org.objectweb.asm.tree.InsnList
import org.objectweb.asm.tree.InsnNode
import org.objectweb.asm.tree.MethodInsnNode
import org.objectweb.asm.tree.MethodNode
import org.objectweb.asm.tree.VarInsnNode

/**
 * Created by ChengTao(chentao.joe@bytedance.com) on 2022/9/7.
 */
internal class LayoutNameToIdRegistryTransformer : ClassTransformer {
    override fun transform(klass: ClassNode) {
        if (klass.name == LAYOUT_NAME_TO_ID_REGISTRY) {
            klass.methods.onEach { method ->
                when {
                    // getLayoutIdByName
                    isGetLayoutIdByNameMethod(method) -> {
                        method.instructions.also { it.clear() }.add(InsnList().apply {
                            add(VarInsnNode(Opcodes.ALOAD, 0))
                            add(
                                MethodInsnNode(
                                    Opcodes.INVOKESTATIC,
                                    LAYOUT_NAME_TO_ID_REGISTRY_IMPL,
                                    METHOD_GET_LAYOUT_ID_BY_NAME,
                                    METHOD_GET_LAYOUT_ID_BY_NAME_DESC,
                                    false
                                )
                            )
                            add(InsnNode(Opcodes.IRETURN))
                        })
                    }

                    // getLayoutNameById
                    isGetLayoutNameByIdMethod(method) -> {
                        method.instructions.also { it.clear() }.add(InsnList().apply {
                            add(VarInsnNode(Opcodes.ILOAD, 0))
                            add(
                                MethodInsnNode(
                                    Opcodes.INVOKESTATIC,
                                    LAYOUT_NAME_TO_ID_REGISTRY_IMPL,
                                    METHOD_GET_LAYOUT_NAME_BY_ID,
                                    METHOD_GET_LAYOUT_NAME_BY_ID_DESC,
                                    false
                                )
                            )
                            add(InsnNode(Opcodes.ARETURN))
                        })
                    }
                }
            }
        }
    }

    private fun isGetLayoutIdByNameMethod(method: MethodNode): Boolean {
        return method.name == METHOD_GET_LAYOUT_ID_BY_NAME
                && method.desc == METHOD_GET_LAYOUT_ID_BY_NAME_DESC
    }

    private fun isGetLayoutNameByIdMethod(method: MethodNode): Boolean {
        return method.name == METHOD_GET_LAYOUT_NAME_BY_ID
                && method.desc == METHOD_GET_LAYOUT_NAME_BY_ID_DESC
    }

    companion object {
        private const val LAYOUT_NAME_TO_ID_REGISTRY =
            "com/bytedance/ultimate/inflater/internal/spi/LayoutNameToIdRegistry"
        private const val LAYOUT_NAME_TO_ID_REGISTRY_IMPL =
            "com/bytedance/ultimate/inflater/internal/spi/LayoutNameToIdRegistryImpl"

        // getLayoutIdByName
        private const val METHOD_GET_LAYOUT_ID_BY_NAME = "getLayoutIdByName"
        private const val METHOD_GET_LAYOUT_ID_BY_NAME_DESC = "(Ljava/lang/String;)I"

        // getLayoutNameById
        private const val METHOD_GET_LAYOUT_NAME_BY_ID = "getLayoutNameById"
        private const val METHOD_GET_LAYOUT_NAME_BY_ID_DESC = "(I)Ljava/lang/String;"
    }
}