/*
 * Copyright (c) 1998-2018 Caucho Technology -- all rights reserved
 *
 * This file is part of Resin(R) Open Source
 *
 * Each copy or derived work must preserve the copyright notice and this
 * notice unmodified.
 *
 * Resin Open Source is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Resin Open Source is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE, or any warranty
 * of NON-INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Resin Open Source; if not, write to the
 *   Free SoftwareFoundation, Inc.
 *   59 Temple Place, Suite 330
 *   Boston, MA 02111-1307  USA
 *
 * @author Scott Ferguson
 *
 * $Id: BodyTagSupport.java,v 1.2 2004/09/29 00:12:48 cvs Exp $
 */

package javax.servlet.jsp.tagext;

import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspWriter;

/**
 * Convenience class defining default behavior for a BodyTag.  Most
 * implementations will extend BodyTagSupport instead of implementing
 * BodyTag directly.
 *
 * <p>The default behavior executes the body once, ignoring the contents.
 */
public class BodyTagSupport extends TagSupport implements BodyTag {
  protected BodyContent bodyContent;

  /**
   * Implementing tags must implement a zero-arg constructor.
   */
  public BodyTagSupport()
  {
  }

  /**
   * Called by the JSP engine just before calling doInitBody().  Empty
   * tags never have setBodyContent called on them.  Because the JSP
   * engine reuses BodyContents, a tag cannot use the body content after
   * doAfterBody completes.
   *
   * @param bodyContent a bodyContent generated by the JSP engine.
   */
  public void setBodyContent(BodyContent bodyContent)
  {
    this.bodyContent = bodyContent;
  }

  /**
   * Returns the tag's body content.  The body content is in scope for
   * the BodyTag methods doInitBody and doAfterBody, but not for the
   * tag methods doStartTag and doEndTag.
   */
  public BodyContent getBodyContent()
  {
    return bodyContent;
  }

  /**
   * Returns the enclosing writer.  For BodyTags with no body, this is
   * equivalent to pageContext.getOut().
   */
  public JspWriter getPreviousOut()
  {
    if (bodyContent != null)
      return bodyContent.getEnclosingWriter();
    else
      return pageContext.getOut();
  }

  /**
   * The default start action is EVAL_BODY_TAG, i.e. evaluate the body.
   */
  public int doStartTag() throws JspException
  {
    return EVAL_BODY_BUFFERED;
  }

  /**
   * The default init action is to do nothing.
   */
  public void doInitBody() throws JspException
  {
  }

  /**
   * Action after the body completes, returning EVAL_BODY_TAG to loop and
   * SKIP_BODY to stop looping.
   *
   * <p>The default action is to only execute the body once.
   */
  public int doAfterBody() throws JspException
  {
    return SKIP_BODY;
  }

  /**
   * Release the tag.
   */
  public void release()
  {
    super.release();
    
    bodyContent = null;
  }
}
