/*
 * Copyright (c) 1998-2018 Caucho Technology -- all rights reserved
 *
 * This file is part of Resin(R) Open Source
 *
 * Each copy or derived work must preserve the copyright notice and this
 * notice unmodified.
 *
 * Resin Open Source is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Resin Open Source is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE, or any warranty
 * of NON-INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Resin Open Source; if not, write to the
 *   Free SoftwareFoundation, Inc.
 *   59 Temple Place, Suite 330
 *   Boston, MA 02111-1307  USA
 *
 * @author Scott Ferguson
 */

package com.caucho.vfs;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Map;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSession;

import com.caucho.util.L10N;

/**
 * The HTTP scheme.  Currently it supports GET and POST.
 *
 * <p>TODO: support WEBDAV, enabling the full Path API.
 */
public class HttpsPath extends HttpPath {
  protected static L10N L = new L10N(HttpsPath.class);

  /**
   * Creates a new HTTP root path with a host and a port.
   *
   * @param host the target host
   * @param port the target port, if zero, uses port 80.
   */
  public HttpsPath(String host, int port)
  {
    super(host, port);
  }

  /**
   * Creates a new HTTP sub path.
   *
   * @param root the HTTP filesystem root
   * @param userPath the argument to the calling lookup()
   * @param newAttributes any attributes passed to http
   * @param path the full normalized path
   * @param query any query string
   */
  public HttpsPath(FilesystemPath root, String userPath,
                   Map<String,Object> newAttributes,
                   String path, String query)
  {
    super(root, userPath, newAttributes, path, query);
  }

  protected HttpPath create(String host, int port)
  {
    return new HttpsPath(host, port);
  }

  protected HttpPath create(FilesystemPath root,
                            String userPath,
                            Map<String,Object> newAttributes,
                            String path, String query)
  {
    return new HttpsPath(root, userPath, newAttributes, path, query);
  }

  /**
   * Returns the scheme, http.
   */
  public String getScheme()
  {
    return "https";
  }
  
  protected int getDefaultPort()
  {
    return 443;
  }

  /**
   * Returns a read stream for a GET request.
   */
  @Override
  public StreamImpl openReadImpl() throws IOException
  {
    /*
    URL url = new URL(getURL());
    
    HttpsURLConnection conn = (HttpsURLConnection) url.openConnection();
    
    
    //conn.setHostnameVerifier(getHostnameVerifier());
    
    InputStream is = conn.getInputStream();
    
    return new VfsStream(is, null);
    */

    HttpStreamWrapper stream = HttpStream.openRead(this);

    stream.setSSL(true);
    
    return stream;
  }

  private HostnameVerifier getHostnameVerifier()
  {
    return new HostnameVerifier() {
      @Override
      public boolean verify(String hostname, SSLSession session)
      {
        return true;
      }
      
    };
  }

  /**
   * Returns a read/write pair for a POST request.
   */
  public StreamImpl openReadWriteImpl() throws IOException
  {
    HttpStreamWrapper stream = HttpStream.openReadWrite(this);

    stream.setSSL(true);
    
    return stream;
  }

  /**
   * Returns a hashCode for the path.
   */
  public int hashCode()
  {
    return 17 + 65537 * super.hashCode() + 37 * _host.hashCode() + _port;
  }

  /**
   * Overrides equals to test for equality with an HTTP path.
   */
  public boolean equals(Object o)
  {
    if (! (o instanceof HttpsPath))
      return false;

    HttpsPath test = (HttpsPath) o;

    if (! _host.equals(test._host))
      return false;
    else if (_port != test._port)
      return false;
    else if (_query != null && ! _query.equals(test._query))
      return false;
    else if (_query == null && test._query != null)
      return false;
    else
      return true;
  }
}
