/*
 * Copyright (c) 1998-2018 Caucho Technology -- all rights reserved
 *
 * This file is part of Resin(R) Open Source
 *
 * Each copy or derived work must preserve the copyright notice and this
 * notice unmodified.
 *
 * Resin Open Source is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Resin Open Source is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE, or any warranty
 * of NON-INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Resin Open Source; if not, write to the
 *
 *   Free Software Foundation, Inc.
 *   59 Temple Place, Suite 330
 *   Boston, MA 02111-1307  USA
 *
 * @author Scott Ferguson
 */

package com.caucho.server.session;

import com.caucho.hessian.io.Hessian2Output;
import com.caucho.hessian.io.HessianDebugOutputStream;
import com.caucho.hessian.io.SerializerFactory;

import java.io.OutputStream;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Serializer for session data
 */
public class HessianSessionSerializer extends SessionSerializer {
  private static final Logger log
    = Logger.getLogger(HessianSessionSerializer.class.getName());
  
  private Hessian2Output _out;

  public HessianSessionSerializer(OutputStream os)
  {
    this(os, Thread.currentThread().getContextClassLoader());
  }

  public HessianSessionSerializer(OutputStream os,
                                  ClassLoader loader)
  {
    this(os, new SerializerFactory(loader));
  }

  public HessianSessionSerializer(OutputStream os,
                                  SerializerFactory factory)
  {
    if (log.isLoggable(Level.FINEST)) {
      os = new HessianDebugOutputStream(os, log, Level.FINEST);
    }
    
    _out = new Hessian2Output(os);
    _out.setSerializerFactory(factory);
  }

  public void setSerializeCollectionType(boolean isEnable)
  {
    _out.getSerializerFactory().setSendCollectionType(isEnable);
  }
  
  @Override
  public void writeInt(int v)
    throws IOException
  {
    _out.writeInt(v);
  }

  @Override
  public void writeObject(Object v)
    throws IOException
  {
    _out.writeObject(v);
  }

  @Override
  public void flush()
  {
    Hessian2Output out = _out;

    if (out != null) {
      try {
        out.flush();
      } catch (IOException e) {
        log.log(Level.FINEST, e.toString(), e);
      }
    }
  }

  @Override
  public void close()
  {
    Hessian2Output out = _out;
    _out = null;

    if (out != null) {
      try {
        out.close();
      } catch (IOException e) {
        log.log(Level.FINEST, e.toString(), e);
      }
    }
  }
}
