package com.cedarsoftware.io.reflect;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ReflectPermission;
import java.lang.reflect.Type;
import java.util.Arrays;

import com.cedarsoftware.io.JsonIoException;
import com.cedarsoftware.util.Converter;
import com.cedarsoftware.util.ReflectionUtils;
import com.cedarsoftware.util.StringUtilities;

/**
 * High-performance field injection utility that automatically adapts to different JDK versions
 * for optimal performance and compatibility.
 * 
 * <p>This class uses the {@code java.version} system property to automatically detect the
 * JDK version and select the most appropriate field injection strategy:</p>
 * 
 * <ul>
 * <li><strong>JDK 8-16:</strong> Uses {@code Field.set()} for final fields and {@code MethodHandle} for regular fields</li>
 * <li><strong>JDK 17+:</strong> Uses {@code VarHandle} for improved performance and module system compatibility</li>
 * </ul>
 * 
 * <p>The JDK version detection and strategy selection is completely automatic and requires no
 * user configuration. This ensures optimal performance across all supported JDK versions while
 * maintaining compatibility with the module system introduced in JDK 9+.</p>
 * 
 * <h3>System Properties Used:</h3>
 * <ul>
 * <li>{@code java.version} - Automatically detected by the JVM to determine injection strategy</li>
 * </ul>
 * 
 * @author Ken Partlow (kpartlow@gmail.com)
 *         John DeRegnaucourt (jereg@gmail.com)
 *         <br>
 *         Copyright (c) Cedar Software LLC
 *         <br><br>
 *         Licensed under the Apache License, Version 2.0 (the "License");
 *         you may not use this file except in compliance with the License.
 *         You may obtain a copy of the License at
 *         <br><br>
 *         <a href="http://www.apache.org/licenses/LICENSE-2.0">License</a>
 *         <br><br>
 *         Unless required by applicable law or agreed to in writing, software
 *         distributed under the License is distributed on an "AS IS" BASIS,
 *         WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *         See the License for the specific language governing permissions and
 *         limitations under the License.
 */
public class Injector {

    private static final boolean IS_JDK17_OR_HIGHER;
    private static final Object LOOKUP;
    private static final Method PRIVATE_LOOKUP_IN_METHOD;
    private static final Method FIND_VAR_HANDLE_METHOD;
    private static final MethodHandle VAR_HANDLE_SET_METHOD;
    private static final Class<?> VAR_HANDLE_CLASS;        // although appears unused, it is intentional for caching

    static {
        int javaVersion = determineJdkMajorVersion();
        IS_JDK17_OR_HIGHER = javaVersion >= 17;

        Object lookup = null;
        Method privateLookupInMethod = null;
        Method findVarHandleMethod = null;
        MethodHandle varHandleSetMethod = null;
        Class<?> varHandleClass = null;

        if (javaVersion >= 9) {
            try {
                Class<?> methodHandlesClass = Class.forName("java.lang.invoke.MethodHandles");
                Class<?> lookupClass = Class.forName("java.lang.invoke.MethodHandles$Lookup");

                Method lookupMethod = ReflectionUtils.getMethod(methodHandlesClass, "lookup");
                lookup = lookupMethod.invoke(null);

                privateLookupInMethod = ReflectionUtils.getMethod(methodHandlesClass,
                        "privateLookupIn", Class.class, lookupClass);

                varHandleClass = Class.forName("java.lang.invoke.VarHandle");
                findVarHandleMethod = ReflectionUtils.getMethod(lookupClass,
                        "findVarHandle", Class.class, String.class, Class.class);
                MethodType setType = MethodType.methodType(void.class, Object.class, Object.class);
                varHandleSetMethod = MethodHandles.publicLookup().findVirtual(varHandleClass, "set", setType);
            } catch (Exception e) {
                // VarHandle reflection setup failed.
                lookup = null;
                privateLookupInMethod = null;
                findVarHandleMethod = null;
                varHandleSetMethod = null;
                varHandleClass = null;
            }
        }

        LOOKUP = lookup;
        PRIVATE_LOOKUP_IN_METHOD = privateLookupInMethod;
        FIND_VAR_HANDLE_METHOD = findVarHandleMethod;
        VAR_HANDLE_SET_METHOD = varHandleSetMethod;
        VAR_HANDLE_CLASS = varHandleClass;
    }

    private final Field field;
    private final String displayName;
    private final String uniqueFieldName;
    private MethodHandle injector;
    private Object varHandle; // For JDK 17+ VarHandle-based injection
    private final boolean useFieldSet; // flag to use Field.set() instead of MethodHandle

    // Constructor for MethodHandle injection
    public Injector(Field field, MethodHandle handle, String uniqueFieldName, String displayName) {
        this.field = field;
        this.displayName = displayName;
        this.uniqueFieldName = uniqueFieldName;
        this.injector = handle;
        this.useFieldSet = false;
    }

    // Constructor for Field.set() fallback injection
    private Injector(Field field, String uniqueFieldName, String displayName, boolean useFieldSet) {
        this.field = field;
        this.displayName = displayName;
        this.uniqueFieldName = uniqueFieldName;
        this.useFieldSet = useFieldSet;
    }

    // Constructor for VarHandle-based injection (JDK 17+)
    private Injector(Field field, Object varHandle, String uniqueFieldName, String displayName) {
        this.field = field;
        this.displayName = displayName;
        this.uniqueFieldName = uniqueFieldName;
        this.varHandle = varHandle;
        this.useFieldSet = false;
    }

    public static Injector create(Field field, String uniqueFieldName) {
        // Security: Validate input parameters
        if (field == null) {
            throw new JsonIoException("Field cannot be null");
        }
        if (uniqueFieldName == null || uniqueFieldName.trim().isEmpty()) {
            throw new JsonIoException("Unique field name cannot be null or empty");
        }
        
        // Security: Check if field access is allowed in secure environments
        String fieldName = field.getName();
        Class<?> declaringClass = field.getDeclaringClass();
        
        // Security: Validate field access permissions
        try {
            SecurityManager sm = System.getSecurityManager();
            if (sm != null) {
                sm.checkPermission(new ReflectPermission("suppressAccessChecks"));
            }
        } catch (SecurityException e) {
            throw new JsonIoException("Security policy denies field access to: " + fieldName + " in class: " + declaringClass.getName(), e);
        }
        
        // Always try to make the field accessible, regardless of whether it is static.
        if (!field.isAccessible()) {
            try {
                // Security: Log the setAccessible call for audit purposes
                field.setAccessible(true);
            } catch (Exception ioe) {
                // Security: Handle access denial gracefully
                if (IS_JDK17_OR_HIGHER) {
                    Injector varHandleInjector = createWithVarHandle(field, uniqueFieldName);
                    if (varHandleInjector != null) {
                        return varHandleInjector;
                    }
                }
                // Final fallback to Field.set() injection
                return new Injector(field, uniqueFieldName, field.getName(), true);
            }
        }

        // For JDK 8-16, if the field is final, use Field.set() (and try to remove the final modifier)
        boolean isFinal = Modifier.isFinal(field.getModifiers());
        if (isFinal && !IS_JDK17_OR_HIGHER) {
            // Security: Validate that final modifier removal is allowed
            try {
                SecurityManager sm = System.getSecurityManager();
                if (sm != null) {
                    sm.checkPermission(new ReflectPermission("suppressAccessChecks"));
                }
            } catch (SecurityException e) {
                throw new JsonIoException("Security policy denies final field modification for: " + fieldName + " in class: " + declaringClass.getName(), e);
            }
            
            try {
                // Security: Be cautious when modifying final fields
                Field modifiersField = ReflectionUtils.getField(Field.class, "modifiers");
                if (modifiersField == null) {
                    throw new JsonIoException("Unable to access modifiers field - possible security restriction");
                }
                modifiersField.setInt(field, field.getModifiers() & ~Modifier.FINAL);
            } catch (Exception ex) {
                // Security: Provide descriptive error message for audit trail
                throw new JsonIoException("Failed to remove final modifier from field: " + fieldName + " in class: " + declaringClass.getName() + ". This may be due to security restrictions.", ex);
            }
            return new Injector(field, uniqueFieldName, field.getName(), true);
        }

        try {
            MethodHandle handle = MethodHandles.lookup().unreflectSetter(field);
            return new Injector(field, handle, uniqueFieldName, field.getName());
        } catch (IllegalAccessException e) {
            if (IS_JDK17_OR_HIGHER) {
                Injector varHandleInjector = createWithVarHandle(field, uniqueFieldName);
                if (varHandleInjector != null) {
                    return varHandleInjector;
                }
            }
            // Fallback to Field.set() injection if we cannot get a MethodHandle.
            return new Injector(field, uniqueFieldName, field.getName(), true);
        }
    }

    private static Injector createWithVarHandle(Field field, String uniqueFieldName) {
        // Security: Validate VarHandle infrastructure is available
        if (PRIVATE_LOOKUP_IN_METHOD == null || FIND_VAR_HANDLE_METHOD == null ||
                VAR_HANDLE_SET_METHOD == null || LOOKUP == null) {
            return null; // Return null to allow fallback to Field.set()
        }

        // Security: Validate access permissions for VarHandle creation (only when SecurityManager is present)
        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            try {
                sm.checkPermission(new ReflectPermission("suppressAccessChecks"));
            } catch (SecurityException e) {
                return null; // Return null to allow fallback to Field.set()
            }
        }

        try {
            // Security: Validate field declaring class to prevent unauthorized access
            Class<?> declaringClass = field.getDeclaringClass();
            if (declaringClass == null) {
                return null;
            }
            
            Object privateLookup = PRIVATE_LOOKUP_IN_METHOD.invoke(null, declaringClass, LOOKUP);
            if (privateLookup == null) {
                return null;
            }
            
            Object varHandle = FIND_VAR_HANDLE_METHOD.invoke(privateLookup, declaringClass,
                    field.getName(), field.getType());
            if (varHandle == null) {
                return null;
            }

            return new Injector(field, varHandle, uniqueFieldName, field.getName());
        } catch (Exception e) {
            // Security: Log but don't fail hard - allow fallback to Field.set()
            return null;
        }
    }

    public static Injector create(Field field, String methodName, String uniqueFieldName) {
        // Security: Validate input parameters
        if (field == null) {
            throw new JsonIoException("Field cannot be null");
        }
        if (methodName == null || methodName.trim().isEmpty()) {
            throw new JsonIoException("Method name cannot be null or empty");
        }
        if (uniqueFieldName == null || uniqueFieldName.trim().isEmpty()) {
            throw new JsonIoException("Unique field name cannot be null or empty");
        }
        
        // Security: Validate method access permissions
        try {
            SecurityManager sm = System.getSecurityManager();
            if (sm != null) {
                sm.checkPermission(new ReflectPermission("suppressAccessChecks"));
            }
        } catch (SecurityException e) {
            throw new JsonIoException("Security policy denies method access for: " + methodName + " in class: " + field.getDeclaringClass().getName(), e);
        }
        
        try {
            MethodType methodType = MethodType.methodType(void.class, field.getType());
            MethodHandle handle = MethodHandles.lookup().findVirtual(field.getDeclaringClass(), methodName, methodType);
            return new Injector(field, handle, uniqueFieldName, methodName);
        } catch (NoSuchMethodException e) {
            throw new JsonIoException("Method not found: " + methodName + " in class: " + field.getDeclaringClass().getName(), e);
        } catch (IllegalAccessException e) {
            throw new JsonIoException("Access denied to method: " + methodName + " in class: " + field.getDeclaringClass().getName(), e);
        }
    }

    public void inject(Object object, Object value) {
        // Security: Validate input parameters
        if (object == null) {
            throw new JsonIoException("Attempting to set field: " + getName() + " on null object.");
        }
        
        // Security: Validate that the object is an instance of the field's declaring class
        Class<?> declaringClass = field.getDeclaringClass();
        if (!declaringClass.isInstance(object)) {
            throw new JsonIoException("Object is not an instance of the field's declaring class. Expected: " + 
                declaringClass.getName() + ", Actual: " + object.getClass().getName() + 
                " for field: " + getName());
        }
        
        // Security: Additional validation for system classes
        String className = declaringClass.getName();
        if (className.startsWith("java.lang.") || className.startsWith("java.security.")) {
            // Extra caution when dealing with core system classes
            SecurityManager sm = System.getSecurityManager();
            if (sm != null) {
                try {
                    sm.checkPermission(new ReflectPermission("suppressAccessChecks"));
                } catch (SecurityException e) {
                    throw new JsonIoException("Security policy denies injection into system class: " + className + " field: " + getName(), e);
                }
            }
        }

        try {
            if (varHandle != null) {
                // Use VarHandle-based injection if available (JDK 17+)
                injectWithVarHandle(object, value);
            } else if (useFieldSet) {
                // For JDK 8-16, fallback to using Field.set()
                field.set(object, value);
            } else {
                // Otherwise, use the MethodHandle-based injection.
                injector.invoke(object, value);
            }
        } catch (ClassCastException e) {
            // Cache field type to avoid repeated getType() calls
            final Class<?> fieldType = field.getType();
            final String fieldName = getName();
            final String displayName = getDisplayName();
            
            String msg = e.getMessage();
            if (StringUtilities.hasContent(msg) && msg.contains("LinkedHashMap")) {
                throw new JsonIoException("Unable to set field: " + fieldName + " using " + displayName + ".", e);
            }
            try {
                Object convertedValue = Converter.convert(value, fieldType);
                if (varHandle != null) {
                    injectWithVarHandle(object, convertedValue);
                } else if (useFieldSet) {
                    field.set(object, convertedValue);
                } else {
                    injector.invoke(object, convertedValue);
                }
            } catch (Throwable t) {
                throw new JsonIoException("Unable to set field: " + fieldName + " using " + displayName + ". Getting a ClassCastException.", e);
            }
        } catch (Throwable t) {
            if (t instanceof JsonIoException) {
                throw (JsonIoException) t;
            }
            throw new JsonIoException("Unable to set field: " + getName() + " using " + getDisplayName(), t);
        }
    }

    private void injectWithVarHandle(Object object, Object value) throws Throwable {
        // Security: Validate VarHandle infrastructure
        if (varHandle == null || VAR_HANDLE_SET_METHOD == null) {
            throw new JsonIoException("Unable to set field: " + getName() + " - VarHandle not available");
        }
        
        // Security: Validate arguments before VarHandle invocation
        if (object == null) {
            throw new JsonIoException("Cannot inject into null object using VarHandle for field: " + getName());
        }
        
        try {
            // Security: Use secure argument list creation
            Object[] args = {varHandle, object, value};
            VAR_HANDLE_SET_METHOD.invokeWithArguments(Arrays.asList(args));
        } catch (Exception e) {
            throw new JsonIoException("VarHandle injection failed for field: " + getName() + " in class: " + field.getDeclaringClass().getName(), e);
        }
    }

    public Class<?> getType() {
        return field.getType();
    }

    public String getName() {
        return field.getName();
    }

    public Type getGenericType() {
        return field.getGenericType();
    }

    public String getDisplayName() {
        return displayName;
    }

    public String getUniqueFieldName() {
        return uniqueFieldName;
    }

    //TODO: remove and wire to java-util API for this (3.8.0+)
    private static int determineJdkMajorVersion() {
        try {
            Method versionMethod = ReflectionUtils.getMethod(Runtime.class, "version");
            Object v = versionMethod.invoke(Runtime.getRuntime());
            Method major = ReflectionUtils.getMethod(v.getClass(), "major");
            return (Integer) major.invoke(v);
        } catch (Exception ignore) {
            try {
                String version = System.getProperty("java.version");
                if (version.startsWith("1.")) {
                    return Integer.parseInt(version.substring(2, 3));
                }
                int dot = version.indexOf('.');
                if (dot != -1) {
                    return Integer.parseInt(version.substring(0, dot));
                }
                return Integer.parseInt(version);
            } catch (Exception ignored) {
                try {
                    String spec = System.getProperty("java.specification.version");
                    return spec.startsWith("1.") ? Integer.parseInt(spec.substring(2)) : Integer.parseInt(spec);
                } catch (NumberFormatException e) {
                    return -1;
                }
            }
        }
    }
}
