package com.chain.util;

import java.util.*;

import com.chain.api.*;
import com.chain.exception.*;
import com.chain.http.*;

/**
 * A transaction builder that enables new off-chain rules functionality.
 * Use this class instead of Transaction.Builder to engage the rules
 * engine for your transactions.
 *
 * This transaction builder requires that recipient accounts be present
 * in a special whitelist present in asset tags.
 *
 * The whitelist tag key is "account_whitelist". The value should be
 * formatted as follows:
 *
 * <pre><code>
 * [
 *   {"account_id": "an-account-ID"},
 *   {"account_alias": "an-account-alias"},
 *   ...
 * ]
 * </code></pre>
 */
public class TxBuilderWithRules extends Transaction.Builder {

  /**
   * Builds a batch of transaction templates. Use this instead of
   * <code>Transaction.buildBatch</code> to engage the rules engine
   * for the contents of your batch.
   *
   * @param client client object which makes server requests
   * @param builders list of transaction builders
   * @return a list of transaction templates
   * @throws ChainException
   */
  public static BatchResponse<Transaction.Template> buildBatch(
      Client client, List<Transaction.Builder> builders) throws ChainException {
    List<Transaction.Builder> validBuilders = new ArrayList<>();
    Map<Integer, APIException> txbuilderErrs = new HashMap<>();
    Map<Integer, Integer> newToOldIndex = new HashMap<>();

    for (int i = 0; i < builders.size(); i++) {
      Transaction.Builder b = builders.get(i);
      if (b instanceof TxBuilderWithRules) {
        try {
          ((TxBuilderWithRules) b).validate(client);

          // Only perform this on successful validation.
          newToOldIndex.put(validBuilders.size(), i);
          validBuilders.add(b);
        } catch (WhitelistViolationException e) {
          txbuilderErrs.put(i, e);
        }
      }
    }

    BatchResponse<Transaction.Template> orig = Transaction.buildBatch(client, validBuilders);

    Map<Integer, Transaction.Template> successes = new HashMap<>();
    Map<Integer, APIException> errors = new HashMap<>();

    // Add original responses, re-indexed to reflect original indices.
    for (Map.Entry<Integer, Transaction.Template> entry : orig.successesByIndex().entrySet()) {
      successes.put(newToOldIndex.get(entry.getKey()), entry.getValue());
    }

    for (Map.Entry<Integer, APIException> entry : orig.errorsByIndex().entrySet()) {
      errors.put(newToOldIndex.get(entry.getKey()), entry.getValue());
    }

    // Add rule violation errors
    for (Map.Entry<Integer, APIException> entry : txbuilderErrs.entrySet()) {
      errors.put(entry.getKey(), entry.getValue());
    }

    return new BatchResponse<Transaction.Template>(successes, errors);
  }

  /**
   * Constructs a single transaction template.
   * @param client client object which makes requests to the server
   * @return a transaction template
   * @throws ChainException
   */
  @Override
  public Transaction.Template build(Client client) throws ChainException {
    validate(client);
    return super.build(client);
  }

  /**
   * A special case of APIException that indicates that an asset is
   * being received by an account that is not included in the
   * asset's whitelist.
   */
  public static class WhitelistViolationException extends APIException {
    WhitelistViolationException(Asset asset, Account account) {
      super(
          "Account \""
              + account.alias
              + "\" ("
              + account.id
              + ") "
              + " is not a whitelisted recipient for "
              + "asset \""
              + asset.alias
              + "\" ("
              + asset.id
              + ")",
          "");
    }
  }

  protected static final String whitelistTag = "account_whitelist";

  protected void validate(Client client) throws ChainException {
    List<Transaction.Action.ControlWithAccount> controlActions = new ArrayList<>();
    for (Transaction.Action action : actions) {
      if (action instanceof Transaction.Action.ControlWithAccount) {
        controlActions.add((Transaction.Action.ControlWithAccount) action);
      }
    }

    // early out if there's nothing to check
    if (controlActions.isEmpty()) {
      return;
    }

    Set<String> assetIds = new HashSet<>();
    Set<String> assetAliases = new HashSet<>();
    Set<String> accountIds = new HashSet<>();
    Set<String> accountAliases = new HashSet<>();

    for (Transaction.Action.ControlWithAccount action : controlActions) {
      if (action.containsKey("asset_id")) {
        assetIds.add((String) action.get("asset_id"));
      } else if (action.containsKey("asset_alias")) {
        assetAliases.add((String) action.get("asset_alias"));
      }

      if (action.containsKey("account_id")) {
        accountIds.add((String) action.get("account_id"));
      } else if (action.containsKey("account_alias")) {
        accountAliases.add((String) action.get("account_alias"));
      }
    }

    List<Asset> assets =
        Helper.getAssetsByIdOrAlias(
            client, new ArrayList<String>(assetIds), new ArrayList<String>(assetAliases));

    List<Account> accounts =
        Helper.getAccountsByIdOrAlias(
            client, new ArrayList<String>(accountIds), new ArrayList<String>(accountAliases));

    Map<String, Asset> assetsById = new HashMap<>();
    Map<String, Asset> assetsByAlias = new HashMap<>();
    Map<String, Account> accountsById = new HashMap<>();
    Map<String, Account> accountsByAlias = new HashMap<>();

    for (Asset a : assets) {
      assetsById.put(a.id, a);
      if (a.alias.length() > 0) {
        assetsByAlias.put(a.alias, a);
      }
    }

    for (Account a : accounts) {
      accountsById.put(a.id, a);
      if (a.alias.length() > 0) {
        accountsByAlias.put(a.alias, a);
      }
    }

    for (Transaction.Action.ControlWithAccount action : controlActions) {
      Asset asset = null;
      if (action.containsKey("asset_id")) {
        asset = assetsById.get((String) action.get("asset_id"));
      } else if (action.containsKey("asset_alias")) {
        asset = assetsByAlias.get((String) action.get("asset_alias"));
      }

      Account account = null;
      if (action.containsKey("account_id")) {
        account = accountsById.get((String) action.get("account_id"));
      } else if (action.containsKey("account_alias")) {
        account = accountsByAlias.get((String) action.get("account_alias"));
      }

      // Let the build API call fail if either of the asset or account is
      // badly-specified.
      if (asset == null || account == null) {
        continue;
      }

      // Early-out if no tags are specified.
      if (asset.tags == null || asset.tags.isEmpty()) {
        continue;
      }

      Object tagVal = asset.tags.get(whitelistTag);

      try {
        boolean accountFound = false;
        List<Map<String, String>> whitelist = (List<Map<String, String>>) tagVal;
        for (Map<String, String> entry : whitelist) {
          if (account.id.equals(entry.get("account_id"))
              || (account.alias != null && account.alias.equals(entry.get("account_alias")))) {
            accountFound = true;
            break;
          }
        }

        if (!accountFound) {
          throw new WhitelistViolationException(asset, account);
        }
      } catch (ClassCastException e) {
        // Continue if the whitelist tags are malformed.
        continue;
      }
    }
  }

  public static class Helper {
    public static List<Asset> getAssetsByIdOrAlias(
        Client client, List<String> assetIds, List<String> assetAliases) throws ChainException {
      int n = 1;
      List<String> toks = new ArrayList<>();
      List<String> params = new ArrayList<>();
      for (String id : assetIds) {
        toks.add("id=$" + n++);
        params.add(id);
      }

      for (String alias : assetAliases) {
        toks.add("alias=$" + n++);
        params.add(alias);
      }

      String filter = "";
      for (int i = 0; i < toks.size(); i++) {
        if (i > 0) {
          filter += " OR ";
        }
        filter += toks.get(i);
      }

      Asset.Items assets =
          new Asset.QueryBuilder().setFilter(filter).setFilterParameters(params).execute(client);

      Set<String> ids = new HashSet<>();
      List<Asset> res = new ArrayList<>();

      while (assets.hasNext()) {
        Asset a = assets.next();
        if (ids.contains(a.id)) {
          continue;
        }
        ids.add(a.id);
        res.add(a);
      }

      return res;
    }

    public static List<Account> getAccountsByIdOrAlias(
        Client client, List<String> accountIds, List<String> accountAliases) throws ChainException {
      int n = 1;
      List<String> toks = new ArrayList<>();
      List<String> params = new ArrayList<>();
      for (String id : accountIds) {
        toks.add("id=$" + n++);
        params.add(id);
      }

      for (String alias : accountAliases) {
        toks.add("alias=$" + n++);
        params.add(alias);
      }

      String filter = "";
      for (int i = 0; i < toks.size(); i++) {
        if (i > 0) {
          filter += " OR ";
        }
        filter += toks.get(i);
      }

      Account.Items accounts =
          new Account.QueryBuilder().setFilter(filter).setFilterParameters(params).execute(client);

      Set<String> ids = new HashSet<>();
      List<Account> res = new ArrayList<>();

      while (accounts.hasNext()) {
        Account a = accounts.next();
        if (ids.contains(a.id)) {
          continue;
        }
        ids.add(a.id);
        res.add(a);
      }

      return res;
    }
  }
}
