/**
 * Copyright (C) 2017 WhiteSource Ltd.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.whitesource.agent.dependency.resolver.bower;
import org.whitesource.agent.Constants;
import org.whitesource.agent.api.model.DependencyInfo;
import org.whitesource.agent.api.model.DependencyType;
import org.whitesource.agent.dependency.resolver.BomFile;
import org.whitesource.agent.dependency.resolver.npm.NpmDependencyResolver;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;


/**
 * Dependency Resolver for BOWER projects.
 *
 * @author eugen.horovitz
 */
public class BowerDependencyResolver extends NpmDependencyResolver {

    /* --- Static Members --- */

    private static final String BOWER_JSON = "bower.json";
    public static final String WS_BOWER_FILE2 = "**/*" + "ws_bower.json";
    public static final String WS_BOWER_FILE1 = "**/*" + "ws-log-response-bower.json";

    /* --- Members --- */

    private final BowerBomParser bomParser;
    private final BowerLsJsonDependencyCollector bomCollector;

    /* --- Constructor --- */

    public BowerDependencyResolver(long npmTimeoutDependenciesCollector, boolean runPreStep) {
        super(runPreStep, null);
        bomParser = new BowerBomParser();
        bomCollector = new BowerLsJsonDependencyCollector(npmTimeoutDependenciesCollector);
    }

    /* --- Overridden methods --- */

    @Override
    protected BowerLsJsonDependencyCollector getDependencyCollector() {
        return bomCollector;
    }

    @Override
    public String[] getBomPattern() {
        return new String[]{Constants.PATTERN + BOWER_JSON};
    }

    @Override
    public String getPreferredFileName() {
        return Constants.DOT + BOWER_JSON;
    }

    @Override
    protected BowerBomParser getBomParser() {
        return bomParser;
    }

    @Override
    protected void enrichDependency(DependencyInfo dependency, BomFile packageJson, String npmAccessToken) {
        dependency.setGroupId(packageJson.getName());
        dependency.setArtifactId(packageJson.getName());
        dependency.setVersion(packageJson.getVersion());
        dependency.setSystemPath(packageJson.getLocalFileName());
        dependency.setFilename(packageJson.getFileName());
        dependency.setDependencyType(getDependencyType());
    }

    @Override
    protected Collection<String> getLanguageExcludes() {
        // exclude files generated by the WhiteSource Bower plugin
        Set<String> excludes = new HashSet<>();
        excludes.add(WS_BOWER_FILE2);
        excludes.add(WS_BOWER_FILE1);
        return excludes;
    }

    @Override
    public Collection<String> getSourceFileExtensions() {
        return Arrays.asList(Constants.JS_EXTENSION);
    }

    @Override
    protected DependencyType getDependencyType() {
        return DependencyType.BOWER;
    }

    @Override
    protected String getDependencyTypeName() {
        return DependencyType.BOWER.name();
    }

    @Override
    protected boolean isMatchChildDependency(DependencyInfo childDependency, String name, String version) {
        return childDependency.getGroupId().equals(name) && childDependency.getVersion().equals(version);
    }


}