package org.whitesource.agent.dependency.resolver.js;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.whitesource.agent.Constants;
import org.whitesource.agent.api.model.DependencyInfo;
import org.whitesource.agent.api.model.DependencyType;
import org.whitesource.agent.dependency.resolver.AbstractDependencyResolver;
import org.whitesource.agent.dependency.resolver.ResolutionResult;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class JavascriptDependencyResolver extends AbstractDependencyResolver {

    private final Logger log = LoggerFactory.getLogger(JavascriptDependencyResolver.class);

    public static final List<String> jsTypeExtensions = Arrays.asList(Constants.JS);

    public JavascriptDependencyResolver() {
    }

    @Override
    protected ResolutionResult resolveDependencies(String projectFolder, String topLevelFolder, Set<String> bomFiles) throws FileNotFoundException {
        Collection<DependencyInfo> dependencies = new LinkedList<>();
        for (String jsFile : bomFiles) {
            try {
                DependencyInfo dependencyInfo = null;
                for (JsLibsRegex jsLib : JsLibsRegex.values()) {
                    if (jsLib.getFileNameRegex() != null) {
                        for (String regex : jsLib.getFileNameRegex()) {
                            dependencyInfo = collectDependency(jsFile, jsFile, jsLib.getLibName(), regex, true);
                            if (dependencyInfo != null) {
                                dependencies.add(dependencyInfo);
                                break;
                            }
                        }
                        if (dependencyInfo != null) {
                            break;
                        }
                    }
                }
                if (dependencyInfo != null) {
                    continue;
                }

                String jsStr = readFile(jsFile, StandardCharsets.UTF_8);
                for (JsLibsRegex jsLib : JsLibsRegex.values()) {
                    if (jsLib.getFileContentRegex() != null) {
                        dependencyInfo = collectDependencyFromContent(jsStr, jsFile, jsLib.getLibName(), jsLib.getFileContentRegex());
                        if (dependencyInfo != null) {
                            dependencies.add(dependencyInfo);
                            break;
                        }
                    }
                }
            } catch (Exception e) {
                log.error("Fail to collect Javascript libraries.");
            }
        }
        return new ResolutionResult(dependencies, getExcludes(), getDependencyType(), topLevelFolder);
    }

    private DependencyInfo collectDependencyFromContent(String str, String filePath, String groupId, List<String> regexList) {
        try {
            for (String regex : regexList) {
                DependencyInfo dependencyInfo = collectDependency(str, filePath, groupId, regex, false);
                if (dependencyInfo != null) {
                    return dependencyInfo;
                }
            }
        } catch (Exception e) {
            log.trace("Fail to match library by content: " + filePath);
        }
        return null;
    }

    private DependencyInfo collectDependency(String str, String filePath, String groupId, String regex, boolean isByName) {
        try {
            DependencyInfo dependencyInfo = new DependencyInfo();
            log.trace("Running regex: " + regex);
            Pattern pattern = Pattern.compile(regex);
            Matcher matcher = pattern.matcher(str);

            matcher.find();
            String version = matcher.group(1);
            String artifactId = isByName ? filePath.substring(filePath.lastIndexOf(File.separator) + 1) : groupId + "-" + version + Constants.JS_EXTENSION;
            if (StringUtils.isEmpty(artifactId) || StringUtils.isEmpty(version) ||
                    (StringUtils.isNotEmpty(version) && !Character.isDigit(version.charAt(0)))) {
                return null;
            }

            dependencyInfo.setGroupId(groupId);
            dependencyInfo.setArtifactId(artifactId);
            dependencyInfo.setFilename(artifactId);
            dependencyInfo.setVersion(version);
            dependencyInfo.setSystemPath(filePath);

            return dependencyInfo;
        } catch (Exception e) {
            log.trace("Fail to match library by regex: " + filePath);
        }
        return null;
    }

    private String readFile(String path, Charset encoding) {
        try {
            byte[] encoded = Files.readAllBytes(Paths.get(path));
            return new String(encoded, encoding);
        } catch (IOException e) {
            log.error("Failed to load file: " + path, e);
        }
        return null;
    }

    @Override
    protected Collection<String> getExcludes() {
        return new ArrayList<>();
    }

    @Override
    protected DependencyType getDependencyType() {
        return null;
    }

    @Override
    protected String getDependencyTypeName() {
        return Constants.JS.toUpperCase();
    }

    @Override
    protected String[] getBomPattern() {
        return new String[]{Constants.PATTERN + Constants.JS_EXTENSION};
    }

    @Override
    protected Collection<String> getLanguageExcludes() {
        return new ArrayList<>();
    }

    @Override
    public Collection<String> getSourceFileExtensions() {
        return jsTypeExtensions;
    }
}
