/**
 * Copyright (C) 2017 WhiteSource Ltd.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.whitesource.agent.dependency.resolver.maven;

import fr.dutra.tools.maven.deptree.core.Node;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.whitesource.agent.Constants;
import org.whitesource.agent.api.model.AgentProjectInfo;
import org.whitesource.agent.api.model.Coordinates;
import org.whitesource.agent.api.model.DependencyInfo;
import org.whitesource.agent.api.model.DependencyType;
import org.whitesource.agent.dependency.resolver.DependencyCollector;
import org.whitesource.agent.hash.ChecksumUtils;
import org.whitesource.agent.utils.CommandLineProcess;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Collect dependencies using 'npm ls' or bower command.
 *
 * @author eugen.horovitz
 */
public class MavenTreeDependencyCollector extends DependencyCollector {

    /* --- Statics Members --- */

    private static final Logger logger = LoggerFactory.getLogger(org.whitesource.agent.dependency.resolver.maven.MavenTreeDependencyCollector.class);

    private static final String MVN_PARAMS_M2PATH_PATH = "help:evaluate";
    private static final String MVN_PARAMS_M2PATH_LOCAL = "-Dexpression=settings.localRepository";
    private static final String MVN_PARAMS_Q = "-q";
    private static final String MVN_PARAMS_FORCE_STDOUT = "-DforceStdout";
    private static final String MVN_PARAMS_TREE = "dependency:tree";
    private static String MVN_COMMAND;
    private static final String SCOPE_TEST = "test";
    private static final String SCOPE_PROVIDED = "provided";
    private static final String USER_HOME = "user.home";
    private static final String M2 = ".m2";
    private static final String REPOSITORY = "repository";
    private static final String ALL = "All";
    private static final String POM = "pom";
    private static final String B_PARAMETER = "-B";
    private final String VERSION_PARAMETER = "-v";
    private static final String FAIL_AT_END = "-fae";
    public static final String TEST_JAR = "test-jar";
    public static final String JAR = "jar";
    private final String MVN_CLEAN = "clean";
    private final String MVN_INSTALL = "install";
    private final String MVN_SKIP_TESTS = "-DskipTests";

    /* --- Members --- */

    protected String M2Path;
    private final Set<String> mavenIgnoredScopes;
    private boolean showMavenTreeError;
    private boolean ignorePomModules;
    private boolean ignoreFailedModules;
    private boolean runPreStep;
    private MavenLinesParser mavenLinesParser;

    /* --- Constructors --- */

    public MavenTreeDependencyCollector(String[] mavenIgnoredScopes, boolean ignorePomModules, boolean ignoreFailedModules, boolean runPreStep) {
        mavenLinesParser = new MavenLinesParser();
        this.mavenIgnoredScopes = new HashSet<>();
        if (mavenIgnoredScopes == null) {
            this.mavenIgnoredScopes.add(SCOPE_PROVIDED);
            this.mavenIgnoredScopes.add(SCOPE_TEST);
        } else {
            if (mavenIgnoredScopes.length == 1 && mavenIgnoredScopes[0].equals(ALL)) {
                // do not filter out any scope
            } else {
                Arrays.stream(mavenIgnoredScopes).filter(exclude -> StringUtils.isBlank(exclude))
                        .map(exclude -> this.mavenIgnoredScopes.add(exclude));
            }
        }
        this.ignorePomModules = ignorePomModules;
        this.runPreStep = runPreStep;
        this.ignoreFailedModules = ignoreFailedModules;
    }

    /* --- Public methods --- */

    @Override
    public Collection<AgentProjectInfo> collectDependencies(String rootDirectory) {
        Collection<AgentProjectInfo> projects = new ArrayList<>();
        if (!this.isMavenExist(rootDirectory)) {
            logger.warn("Please install maven");
        } else {
            if (runPreStep) {
                try {
                    CommandLineProcess mvnCleanInstall = new CommandLineProcess(rootDirectory, getCleanInstallCommandParams());
                    mvnCleanInstall.executeProcess();
                    if (mvnCleanInstall.isErrorInProcess()) {
                        logger.warn("Failed to execute the command {}", getCleanInstallCommandParams());
                    }
                } catch (Exception e) {
                    logger.warn("Error while execute dependencies after running {} on {}, {}", getCleanInstallCommandParams(), rootDirectory, e.getMessage());
                    logger.debug("Error: {}", e.getStackTrace());
                }
            }

            if (StringUtils.isBlank(M2Path)) {
                String curM2Path = getMavenM2Path(Constants.DOT);
                this.M2Path = curM2Path != null ? curM2Path : getMavenM2Path("");
            }
            Map<String, List<DependencyInfo>> pathToDependenciesMap = new HashMap<>();
            try {
                CommandLineProcess mvnDependencies = new CommandLineProcess(rootDirectory, getLsCommandParamsBatchMode());
                List<String> lines = mvnDependencies.executeProcess();
                if (mvnDependencies.isErrorInProcess()) {
                    logger.debug("Failed to execute the command {}", getLsCommandParamsBatchMode());
                    if (!ignoreFailedModules) {
                        mvnDependencies = new CommandLineProcess(rootDirectory, getLsCommandParams());
                        lines = mvnDependencies.executeProcess();
                    }
                }
                if (!mvnDependencies.isErrorInProcess() || ignoreFailedModules) {
                    List<Node> nodes = mavenLinesParser.parseLines(lines);

                    logger.info("End parsing pom files , found : " + String.join(Constants.COMMA,
                            nodes.stream().map(node -> node.getArtifactId()).collect(Collectors.toList())));

                    projects = nodes.stream()
                            .filter(node -> !this.ignorePomModules || (ignorePomModules && !node.getPackaging().equals(POM)))
                            .map(tree -> {

                                List<DependencyInfo> dependencies = new LinkedList<>();
                                Stream<Node> nodeStream = tree.getChildNodes().stream().filter(node -> !mavenIgnoredScopes.contains(node.getScope()));
                                dependencies.addAll(nodeStream.map(node -> getDependencyFromNode(node, pathToDependenciesMap)).collect(Collectors.toList()));

                                Map<String, String> pathToSha1Map = pathToDependenciesMap.keySet().stream().distinct().parallel().collect(Collectors.toMap(file -> file, file -> getSha1(file)));
                                pathToSha1Map.entrySet().forEach(pathSha1Pair -> pathToDependenciesMap.get(pathSha1Pair.getKey()).stream().forEach(dependency -> {
                                    dependency.setSha1(pathSha1Pair.getValue());
                                    dependency.setSystemPath(pathSha1Pair.getKey());
                                }));

                                AgentProjectInfo projectInfo = new AgentProjectInfo();
                                projectInfo.setCoordinates(new Coordinates(tree.getGroupId(), tree.getArtifactId(), tree.getVersion()));
                                dependencies.stream().forEach(dependency -> projectInfo.getDependencies().add(dependency));
                                return projectInfo;
                            }).collect(Collectors.toList());
                } else {
                    logger.warn("Failed to scan and send {}", getLsCommandParams());
                }
            } catch (IOException e) {
                logger.warn("Error getting dependencies after running {} on {}, {}", getLsCommandParams(), rootDirectory, e.getMessage());
                logger.debug("Error: {}", e.getStackTrace());
            }

            if (projects != null && projects.isEmpty()) {
                if (!showMavenTreeError) {
                    logger.warn("Failed to getting dependencies after running '{}'", Arrays.toString(getLsCommandParams()));
                    showMavenTreeError = true;
                }
            }
        }
        return projects;
    }

    protected String getSha1(String filePath) {
        try {
            return ChecksumUtils.calculateSHA1(new File(filePath));
        } catch (IOException e) {
            logger.error("Failed getting " + filePath + ". File might not be solved by Checkmarx server.");
            return Constants.EMPTY_STRING;
        }
    }

    private boolean isMavenExist(String rootDirectory) {
        try {
            CommandLineProcess mvnProcess = new CommandLineProcess(rootDirectory, getVersionCommandParams());
            List<String> lines = mvnProcess.executeProcess();
            if (mvnProcess.isErrorInProcess() || lines.isEmpty()) {
                logger.debug("Failed to get maven version");
                return false;
            } else {
                logger.debug("Maven : {}", lines);
                return true;
            }
        } catch (Exception e) {
            logger.warn("Failed to get maven version : {}", e.getMessage());
            return false;
        }
    }

    private static String getMvnCommand() {
        String mvnPath = System.getenv("CX_MAVEN_PATH");
        mvnPath = StringUtils.isNotEmpty(mvnPath) ? mvnPath : System.getProperty("CX_MAVEN_PATH");
        if (StringUtils.isNotEmpty(mvnPath)) {
            mvnPath = mvnPath.endsWith(File.separator) ? mvnPath : mvnPath + File.separator;
            MVN_COMMAND = mvnPath + "mvn";
        } else {
            MVN_COMMAND = "mvn";
        }
        return MVN_COMMAND;
    }

    private DependencyInfo getDependencyFromNode(Node node, Map<String, List<DependencyInfo>> paths) {
        logger.debug("converting node to dependency :" + node.getArtifactId());
        DependencyInfo dependency = new DependencyInfo(node.getGroupId(), node.getArtifactId(), node.getVersion());
        dependency.setDependencyType(DependencyType.MAVEN);
        dependency.setScope(node.getScope());
        dependency.setType(node.getPackaging());

        String shortName;
        if (StringUtils.isBlank(node.getClassifier())) {
            shortName = dependency.getArtifactId() + Constants.DASH + dependency.getVersion() + Constants.DOT + node.getPackaging();
        } else {
            String nodePackaging = node.getPackaging();
            if (nodePackaging.equals(TEST_JAR)) {
                nodePackaging = JAR;
            }
            shortName = dependency.getArtifactId() + Constants.DASH + dependency.getVersion() + Constants.DASH + node.getClassifier() + Constants.DOT + nodePackaging;
        }

        String filePath = Paths.get(M2Path, dependency.getGroupId().replace(Constants.DOT, File.separator), dependency.getArtifactId(), dependency.getVersion(), shortName).toString();
        logger.debug("Adding dependency: " + filePath);
        if (!paths.containsKey(filePath)) {
            paths.put(filePath, new ArrayList<>());
        }
        paths.get(filePath).add(dependency);
        if (StringUtils.isNotBlank(filePath)) {
            File jarFile = new File(filePath);
            if (jarFile.exists()) {
                dependency.setFilename(jarFile.getName());
                logger.trace("Dependency exist under: " + filePath);
            } else {
                logger.trace("Dependency does NOT exist under: " + filePath);
            }
        }

        node.getChildNodes().forEach(childNode -> dependency.getChildren().add(getDependencyFromNode(childNode, paths)));
        return dependency;
    }

    /* --- Private methods --- */

    private String[] getCleanInstallCommandParams() {
        if (isWindows()) {
            return addInstallOptionalParams(new String[]{Constants.CMD, C_CHAR_WINDOWS, getMvnCommand(), MVN_CLEAN, MVN_INSTALL, MVN_SKIP_TESTS});
        } else {
            return addInstallOptionalParams(new String[]{getMvnCommand(), MVN_CLEAN, MVN_INSTALL, MVN_SKIP_TESTS});
        }
    }

    private String[] getVersionCommandParams() {
        if (isWindows()) {
            return new String[]{Constants.CMD, C_CHAR_WINDOWS, getMvnCommand(), VERSION_PARAMETER};
        } else {
            return new String[]{getMvnCommand(), VERSION_PARAMETER};
        }
    }

    private String[] getLsCommandParams() {
        if (isWindows()) {
            return addLsOptionalParams(new String[]{Constants.CMD, C_CHAR_WINDOWS, getMvnCommand(), MVN_PARAMS_TREE});
        } else {
            return addLsOptionalParams(new String[]{getMvnCommand(), MVN_PARAMS_TREE});
        }
    }

    private String[] addLsOptionalParams(String[] command) {
        if (ignoreFailedModules) {
            command = ArrayUtils.add(command, FAIL_AT_END);
        }
        if (StringUtils.isNotEmpty(getMvnLsOptions())){
            command = ArrayUtils.add(command, getMvnLsOptions());
        }

        return command;
    }

    private String[] addInstallOptionalParams(String[] command) {
        String mvnInstallOpt = getMvnInstallOptions();
        if (StringUtils.isNotEmpty(mvnInstallOpt)) {
            return ArrayUtils.add(command, mvnInstallOpt);
        }
        return command;
    }

    private String getMvnLsOptions() {
        String mvnInstallOpt = System.getenv("CX_MVN_LS_OPT");
        return StringUtils.isNotEmpty(mvnInstallOpt) ? mvnInstallOpt : System.getProperty("CX_MVN_LS_OPT");
    }

    private String getMvnInstallOptions() {
        String mvnInstallOpt = System.getenv("CX_MVN_INSTALL_OPT");
        return StringUtils.isNotEmpty(mvnInstallOpt) ? mvnInstallOpt : System.getProperty("CX_MVN_INSTALL_OPT");
    }

    private String[] getLsCommandParamsBatchMode() {
        String[] commandParams = getLsCommandParams();
        String[] result = new String[commandParams.length + 1];
        for (int i = 0; i < commandParams.length; i++) {
            result[i] = commandParams[i];
        }
        result[result.length - 1] = B_PARAMETER;
        return result;
    }

    protected String getMavenM2Path(String rootDirectory) {
        String cxM2Path = System.getenv("CX_M2_PATH");
        cxM2Path = StringUtils.isNotEmpty(cxM2Path) ? cxM2Path : System.getProperty("CX_M2_PATH");
        if (StringUtils.isNotEmpty(cxM2Path)) {
            return cxM2Path;
        }
        String currentUsersHomeDir = System.getProperty(USER_HOME);
        File m2Path = Paths.get(currentUsersHomeDir, M2, REPOSITORY).toFile();

        if (m2Path.exists()) {
            return m2Path.getAbsolutePath();
        }
        String[] params = null;
        if (isWindows()) {
            params = new String[]{Constants.CMD, C_CHAR_WINDOWS, getMvnCommand(), MVN_PARAMS_M2PATH_PATH, MVN_PARAMS_M2PATH_LOCAL, MVN_PARAMS_Q, MVN_PARAMS_FORCE_STDOUT};
        } else {
//            params = new String[]{getMvnCommand(), MVN_PARAMS_M2PATH_PATH, MVN_PARAMS_M2PATH_LOCAL, MVN_PARAMS_Q, MVN_PARAMS_FORCE_STDOUT};
            params = new String[]{getMvnCommand(), MVN_PARAMS_M2PATH_PATH, MVN_PARAMS_M2PATH_LOCAL};
        }
        try {
            CommandLineProcess mvnProcess = new CommandLineProcess(rootDirectory, params);
            List<String> lines = mvnProcess.executeProcess();
            if (!mvnProcess.isErrorInProcess()) {
                Optional<String> pathLine = lines.stream().filter(line -> (new File(line).exists())).findFirst();
                if (pathLine.isPresent()) {
                    return pathLine.get();
                } else {
                    logger.warn("could not get m2 path : {} out: {}", rootDirectory, lines.stream().reduce(Constants.EMPTY_STRING, String::concat));
                    showMavenTreeError = true;
                    return null;
                }
            } else {
                logger.warn("Failed to scan and send {}", getLsCommandParams());
                return null;
            }
        } catch (IOException io) {
            logger.warn("could not get m2 path : {}", io.getMessage());
            showMavenTreeError = true;
            return null;
        }
    }

}
