/**
 * Copyright (C) 2017 WhiteSource Ltd.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.whitesource.agent.dependency.resolver.nuget;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.whitesource.agent.Constants;
import org.whitesource.agent.api.model.DependencyInfo;
import org.whitesource.agent.api.model.DependencyType;
import org.whitesource.agent.dependency.resolver.AbstractDependencyResolver;
import org.whitesource.agent.dependency.resolver.ResolutionResult;
import org.whitesource.agent.dependency.resolver.nuget.packagesConfig.NugetConfigFileType;
import org.whitesource.agent.dependency.resolver.nuget.packagesConfig.NugetPackagesConfigXmlParser;
import org.whitesource.agent.utils.CommandLineProcess;
import org.whitesource.agent.utils.FilesScanner;
import org.whitesource.fs.CommandLineArgs;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

import static org.whitesource.agent.dependency.resolver.dotNet.DotNetRestoreCollector.RESTORE;

/**
 * @author yossi.weinberg
 */
public class NugetDependencyResolver extends AbstractDependencyResolver {

    /* --- Static members --- */

    private final Logger logger = LoggerFactory.getLogger(NugetDependencyResolver.class);
    public static final String CONFIG = ".config";
    public static final String CSPROJ = ".csproj";

    /* --- Members --- */

    private final String whitesourceConfiguration;
    private final String bomPattern;
    private final NugetConfigFileType nugetConfigFileType;
    private boolean nugetRestoreDependencies;

    /* --- Constructor --- */

    public NugetDependencyResolver(String whitesourceConfiguration,
                                   NugetConfigFileType nugetConfigFileType,
                                   boolean nugetRestoreDependencies) {
        super();
        this.nugetRestoreDependencies = nugetRestoreDependencies;
        this.whitesourceConfiguration = whitesourceConfiguration;
        this.nugetConfigFileType = nugetConfigFileType;
        if (this.nugetConfigFileType == NugetConfigFileType.CONFIG_FILE_TYPE) {
            bomPattern = Constants.PATTERN + CONFIG;
        } else {
            bomPattern = Constants.PATTERN + CSPROJ;
        }
    }

    /* --- Overridden methods --- */

    @Override
    protected ResolutionResult resolveDependencies(String projectFolder, String topLevelFolder, Set<String> configFiles) {
        if (StringUtils.isNotEmpty(getNugetExePath())) {
            executeNugetRestore(projectFolder);
        }
        return getResolutionResultFromParsing(topLevelFolder, configFiles, false);
    }

    protected ResolutionResult getResolutionResultFromParsing(String topLevelFolder, Set<String> configFiles, boolean onlyDependenciesFromReferenceTag) {
        Collection<DependencyInfo> dependencies = parseNugetPackageFiles(configFiles, onlyDependenciesFromReferenceTag);
        return new ResolutionResult(dependencies, new LinkedList<>(), getDependencyType(), topLevelFolder);
    }

    protected void executeNugetRestore(String projectFolder) {
        List<String> slnFiles = getSlnFiles(projectFolder);
        logger.info("Nuget .sln files: " + ArrayUtils.toString(slnFiles));
        if (slnFiles != null && slnFiles.size() > 0) {
            executeNugetRestore(projectFolder, getNugetExePath(), slnFiles, getNugetOptions());
        } else {
            logger.warn("No '.sln' files were found under: " + projectFolder);
        }
    }

    @Override
    protected Collection<String> getExcludes() {
        List<String> excludes = new LinkedList<>();
        excludes.add(CommandLineArgs.CONFIG_FILE_NAME);
        return excludes;
    }

    @Override
    public Collection<String> getSourceFileExtensions() {
        return new ArrayList<>(Arrays.asList(Constants.DLL, Constants.EXE, Constants.NUPKG, Constants.CS));
    }

    @Override
    protected DependencyType getDependencyType() {
        return DependencyType.NUGET;
    }

    @Override
    protected String getDependencyTypeName() {
        return DependencyType.NUGET.name();
    }

    @Override
    protected String[] getBomPattern() {
        return new String[]{this.bomPattern};
    }

    @Override
    protected Collection<String> getLanguageExcludes() {
        return new ArrayList<>();
    }

    protected Collection<DependencyInfo> parseNugetPackageFiles(Set<String> configFilesPath, boolean getDependenciesFromReferenceTag) {
        // get configuration file path
        Set<DependencyInfo> dependencies = new HashSet<>();
        for (String configFilePath : configFilesPath) {
            // don't scan the whitesource configuration file
            // sometimes FSA is called from outside and there is no config file
            if (whitesourceConfiguration == null || !new File(whitesourceConfiguration).getAbsolutePath().equals(configFilePath)) {
                File configFile = new File(configFilePath);
                // check filename again (just in case)
                if (!configFile.getName().equals(CommandLineArgs.CONFIG_FILE_NAME)) {
                    NugetPackagesConfigXmlParser parser = new NugetPackagesConfigXmlParser(configFile, this.nugetConfigFileType);
                    Set<DependencyInfo> dependenciesFromSingleFile = parser.parsePackagesConfigFile(getDependenciesFromReferenceTag, configFilePath);
                    if (!dependenciesFromSingleFile.isEmpty()) {
                        dependencies.addAll(dependenciesFromSingleFile);
                    }
                }
            }
        }
        return dependencies;
    }

    public void executeNugetRestore(String folder, String nugetExe, List<String> slnFiles, String nugetOpt) {
        for (String slnFile : slnFiles) {
            String slnFilePath = buildPath(folder, slnFile);

            String[] command = getRestoreParams(nugetExe, slnFilePath, nugetOpt);
            String commandString = String.join(Constants.WHITESPACE, command);
            logger.info("Running command : '{}'", commandString);
            CommandLineProcess NugetRestore = new CommandLineProcess(getNugetRunningDir(folder), command);
            try {
                NugetRestore.executeProcess();
            } catch (IOException e) {
                logger.error("Could not run '{}' in folder: {}", commandString, folder);
            }
            if (!NugetRestore.isErrorInProcess()) {
                logger.info("Successfully finished to run '{}'", commandString);
            } else {
                logger.warn("Could not run '{}' in folder: {}", commandString, folder);
            }
        }
    }

    private String buildPath(String basePath, String filePath) {
        try {
            File file;
            basePath = basePath.replaceAll("/[\\\\|\\/]/", File.separator);
            filePath = filePath.replaceAll("/[\\\\|\\/]/", File.separator);
            file = new File(filePath);
            if (file.exists()) {
                return file.getAbsolutePath();
            }

            filePath = filePath.startsWith(File.separator) ? filePath : File.separator + filePath;
            filePath = basePath + filePath;
            file = new File(filePath);
            if (file.exists()) {
                return file.getAbsolutePath();
            }
            logger.error("Fail to build path for solution file: {}, in folder: {}", filePath, basePath);
            return null;
        } catch (Exception e) {
            logger.error("Fail to build path due to exception for solution file: {}, in folder: {}", filePath, basePath);
            return null;
        }
    }

    protected static String[] getRestoreParams(String nugetExe, String slnFile, String nugetOpt) {
        String[] cmd = new String[]{nugetExe, RESTORE, slnFile, nugetOpt};
        return ArrayUtils.removeElements(cmd, null, null, null);
    }

    protected String getNugetExePath() {
        String nugetPath = StringUtils.isNotEmpty(System.getenv("CX_NUGET_PATH")) ?
                System.getenv("CX_NUGET_PATH") : System.getProperty("CX_NUGET_PATH");

        return StringUtils.isNotEmpty(nugetPath) ? nugetPath.trim() : null;
    }

    protected String getNugetRunningDir(String dir) {
        String runDir = StringUtils.isNotEmpty(System.getenv("CX_NUGET_RUN_DIR")) ?
                System.getenv("CX_NUGET_RUN_DIR") : System.getProperty("CX_NUGET_RUN_DIR");
        if (StringUtils.isEmpty(runDir)) {
            return dir;
        }

        runDir = runDir.trim();
        return runDir.startsWith(File.separator) ? dir + runDir : dir + File.separator + runDir;
    }

    protected List<String> getSlnFiles(String projectFolder) {
        String slnFilesPath = System.getenv("CX_NUGET_SLN_FILES");
        slnFilesPath = StringUtils.isNotEmpty(slnFilesPath) ? slnFilesPath : System.getProperty("CX_NUGET_SLN_FILES");
        slnFilesPath = slnFilesPath.trim();
        if (StringUtils.isNotEmpty(slnFilesPath)) {
            return Arrays.stream(slnFilesPath.split(",")).map(String::trim).
                    map(str -> str = projectFolder + File.separator + str).
                    collect(Collectors.toList());
        }
        return Arrays.stream(new FilesScanner().getDirectoryContent(projectFolder, new String[]{"**/*.sln"}, null, false, false)).
                map(str -> str = projectFolder + File.separator + str).
                collect(Collectors.toList());
    }

    protected String getNugetOptions() {
        String nugetOpt = StringUtils.isNotEmpty(System.getenv("CX_NUGET_OPTIONS")) ?
                System.getenv("CX_NUGET_OPTIONS") : System.getProperty("CX_NUGET_OPTIONS");

        return StringUtils.isNotEmpty(nugetOpt) ? nugetOpt.trim() : null;
    }

}
