package org.whitesource.agent.utils;

import com.sun.jna.Native;
import com.sun.jna.platform.win32.Kernel32;
import org.apache.commons.compress.utils.IOUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.whitesource.agent.Constants;
import org.whitesource.agent.dependency.resolver.DependencyCollector;

import java.io.*;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.*;

/**
 * @author raz.nitzan
 */
public class CommandLineProcess {

    private final Logger logger = LoggerFactory.getLogger(org.whitesource.agent.utils.CommandLineProcess.class);

    private String rootDirectory;
    private String[] args;
    private long timeoutReadLineSeconds;
    private long timeoutProcessMinutes;
    private boolean errorInProcess = false;
    private Process processStart = null;
    private File errorLog = new File("error.log");

    private static int errCount = 1;

    private static final long DEFAULT_TIMEOUT_READLINE_SECONDS = 60;
    private static final long DEFAULT_TIMEOUT_PROCESS_MINUTES = 15;
    private static final String WINDOWS_SEPARATOR = "\\";
    private static final SimpleDateFormat DF = new SimpleDateFormat("MMddHHmmss");
    private static final String OSA_ERROR_FOLDER_PATH = String.format(File.separator + "osa-logs" + File.separator + "scan-%s" + File.separator + "error", DF.format(new Timestamp(System.currentTimeMillis())));
    private static final String OSA_ERROR_FILE = File.separator + "errcmd";
    private static final String OSA_ERROR_FILE_EXT = ".log";

    public CommandLineProcess(String rootDirectory, String[] args) {
        this.rootDirectory = rootDirectory;
        this.args = args;
        this.timeoutReadLineSeconds = initReadlineTimeout();
        this.timeoutProcessMinutes = initProcessTimeout();
    }

    private boolean enrichErrLogs() {
        try {
            String isErrLogs = System.getenv("FSA_ENRICH_ERR_LOGS");
            isErrLogs = StringUtils.isNotEmpty(isErrLogs) ? isErrLogs : System.getProperty("FSA_ENRICH_ERR_LOGS");
            if (StringUtils.isNotEmpty(isErrLogs)) {
                return Boolean.parseBoolean(isErrLogs);
            }
            return false;
        } catch (Exception e) {
            return false;
        }
    }

    public String getErrLogPath() {
        String errDir = System.getenv("FSA_ERR_LOGS_PATH");
        errDir = StringUtils.isNotEmpty(errDir) ? errDir : System.getProperty("FSA_ERR_LOGS_PATH");

        return isDirExist(errDir) ? errDir : System.getProperty("user.dir");
    }

    private boolean isDirExist(String path) {
        if (StringUtils.isNotEmpty(path)) {
            File dir = new File(path);
            try {
                return dir.exists() && dir.isDirectory();
            } catch (Exception e) {
                logger.warn("Fail to set FSA error logs folder on: " + path);
            }
        }
        return false;
    }

    private long initReadlineTimeout() {
        try {
            String readlineTimeout = System.getenv("FSA_READLINE_TIMEOUT");
            readlineTimeout = StringUtils.isNotEmpty(readlineTimeout) ? readlineTimeout : System.getProperty("FSA_READLINE_TIMEOUT");
            if (StringUtils.isNotEmpty(readlineTimeout)) {
                return Long.parseLong(readlineTimeout);
            }
            return DEFAULT_TIMEOUT_READLINE_SECONDS;
        } catch (Exception e) {
            return DEFAULT_TIMEOUT_READLINE_SECONDS;
        }
    }

    private long initProcessTimeout() {
        try {
            String processTimeout = System.getenv("FSA_PROCESS_TIMEOUT");
            processTimeout = StringUtils.isNotEmpty(processTimeout) ? processTimeout : System.getProperty("FSA_PROCESS_TIMEOUT");
            if (StringUtils.isNotEmpty(processTimeout)) {
                return Long.parseLong(processTimeout);
            }
            return DEFAULT_TIMEOUT_PROCESS_MINUTES;
        } catch (Exception e) {
            return DEFAULT_TIMEOUT_PROCESS_MINUTES;
        }
    }

    private boolean isUncPath() {
        String osName = System.getProperty(Constants.OS_NAME);
        if (osName.startsWith(Constants.WINDOWS)) {
            return StringUtils.isNotEmpty(rootDirectory) && rootDirectory.startsWith("\\\\");
        }
        return false;
    }

    public List<String> executeProcess() throws IOException {
        return executeProcess(true, false);
    }

    private boolean isWinCmd() {
        try {
            return args[0].equalsIgnoreCase("cmd") && args[1].equalsIgnoreCase("/c");
        } catch (Exception e) {
            return false;
        }
    }

    private List<String> executeProcess(boolean includeOutput, boolean includeErrorLines) throws IOException {
        List<String> linesOutput = new LinkedList<>();
        ProcessBuilder pb;
        if (isUncPath()) {
            if (isWinCmd()) {
                args = ArrayUtils.add(args, 2, "pushd");
                args = ArrayUtils.add(args, 3, rootDirectory);
                args = ArrayUtils.add(args, 4, "&&");
            } else {
                args = ArrayUtils.add(args, 0, "pushd");
                args = ArrayUtils.add(args, 1, rootDirectory);
                args = ArrayUtils.add(args, 2, "&&");
            }
            args = ArrayUtils.add(args, args.length, "&&");
            args = ArrayUtils.add(args, args.length, "popd");
            pb = new ProcessBuilder(args);
            pb.directory(new File(System.getProperty("user.home")));
        } else {
            pb = new ProcessBuilder(args);
            String osName = System.getProperty(Constants.OS_NAME);
            if (osName.startsWith(Constants.WINDOWS)) {
                rootDirectory = getShortPath(rootDirectory);
            }
            pb.directory(new File(rootDirectory));
        }
        // redirect the error output to avoid output of npm ls by operating system
        String redirectErrorOutput = DependencyCollector.isWindows() ? "nul" : "/dev/null";
        if (includeErrorLines) {
            pb.redirectError(errorLog);
        } else {
            pb.redirectError(new File(redirectErrorOutput));
        }
        if (!includeOutput || includeErrorLines) {
            pb.redirectOutput(new File(redirectErrorOutput));
        }
        if (!includeErrorLines) {
            logger.info("Executing command:");
            logger.info("{}> {}", rootDirectory, String.join(Constants.WHITESPACE, args));
        }
        this.processStart = pb.start();
        if (includeOutput) {
            InputStreamReader inputStreamReader;
            BufferedReader reader;
            ExecutorService executorService = Executors.newFixedThreadPool(1);
            if (!includeErrorLines) {
                inputStreamReader = new InputStreamReader(this.processStart.getInputStream());
            } else {
                inputStreamReader = new InputStreamReader(this.processStart.getErrorStream());
            }
            reader = new BufferedReader(inputStreamReader);
            try {
                this.errorInProcess = readBlock(inputStreamReader, reader, executorService, linesOutput, includeErrorLines);
            } catch (TimeoutException e) {
                this.errorInProcess = true;
                CommandLineErrors.addFailedCmd(Arrays.toString(args));
                this.processStart.destroy();
                return linesOutput;
            }
        }
        try {
            this.processStart.waitFor(this.timeoutProcessMinutes, TimeUnit.MINUTES);
        } catch (InterruptedException e) {
            this.errorInProcess = true;
            CommandLineErrors.addFailedCmd(Arrays.toString(args));
            logger.error("'{}' was interrupted {}", args, e);
        }
        if (this.processStart.isAlive() && errorInProcess) {
            logger.warn("Error executing command destroying process");
            this.processStart.destroy();
            return linesOutput;
        }
        if (this.getExitStatus() != 0) {
            File dir = new File(this.rootDirectory);
            int numOfFiles = dir.exists() ? FileUtils.listFiles(dir, null, true).size() : -1;
            logger.error("Execute command finished unexpected, exit code: '{}', directory: {}, is directory valid: {}, folder content size: {}, Output:",
                    this.getExitStatus(), this.rootDirectory, dir.exists(), numOfFiles);
            if (enrichErrLogs()) {
                logErrors(linesOutput);
                errCount++;
            }
            this.errorInProcess = true;
            CommandLineErrors.addFailedCmd(Arrays.toString(args));
        }
//        printErrors();
        return linesOutput;
    }

    private void logErrors(List<String> lines) {
        try {
            File errDir = new File(getErrLogPath() + OSA_ERROR_FOLDER_PATH);
            if (!errDir.exists()) {
                errDir.mkdirs();
            }
            String filePath;
            if (this.errCount == 1) {
                filePath = errDir.getAbsolutePath() + OSA_ERROR_FILE + OSA_ERROR_FILE_EXT;
            } else {
                filePath = errDir.getAbsolutePath() + OSA_ERROR_FILE + this.errCount + OSA_ERROR_FILE_EXT;
            }
            logger.info("Failed command: '" + Arrays.toString(args) + "', error log: " + filePath);
            try (FileWriter writer = new FileWriter(filePath)) {
                writer.write("##### Failed command #####" + System.lineSeparator());
                writer.write(Arrays.toString(args) + System.lineSeparator());
                writer.write("##########################" + System.lineSeparator() + System.lineSeparator());

                for (String line : lines) {
                    writer.write(line + System.lineSeparator());
                }
            }
        } catch (Exception e) {
            logger.warn("Fail to create error log for command: " + Arrays.toString(args));
        }
    }

    // using this technique to print to the log the Process's errors as it the easiest way i found to do so -
    // ues a file to redirect the errors to, read from it and then delete it.
    // if you find a better way - go ahead and replace it
    private void printErrors() {
        if (errorLog.isFile()) {
            FileReader fileReader;
            try {
                fileReader = new FileReader(errorLog);
                BufferedReader bufferedReader = new BufferedReader(fileReader);
                String currLine;
                while ((currLine = bufferedReader.readLine()) != null) {
                    logger.warn(currLine);
                }
                fileReader.close();
            } catch (Exception e) {
                logger.warn("Error printing cmd command errors {} ", e.getMessage());
                logger.error("Error: {}", e.getStackTrace());
            } finally {
                try {
                    FileUtils.forceDelete(errorLog);
                } catch (IOException e) {
                    logger.warn("Error closing cmd command errors file {} ", e.getMessage());
                    logger.error("Error: {}", e.getStackTrace());
                }
            }
        }
    }

    //get windows short path
    private String getShortPath(String rootPath) {
        File file = new File(rootPath);
        String lastPathAfterSeparator = null;
        String shortPath = getWindowsShortPath(file.getAbsolutePath());
        if (StringUtils.isNotEmpty(shortPath)) {
            return getWindowsShortPath(file.getAbsolutePath());
        } else {
            while (StringUtils.isEmpty(getWindowsShortPath(file.getAbsolutePath()))) {
                String filePath = file.getAbsolutePath();
                if (StringUtils.isNotEmpty(lastPathAfterSeparator)) {
                    lastPathAfterSeparator = file.getAbsolutePath().substring(filePath.lastIndexOf(WINDOWS_SEPARATOR), filePath.length()) + lastPathAfterSeparator;
                } else {
                    lastPathAfterSeparator = file.getAbsolutePath().substring(filePath.lastIndexOf(WINDOWS_SEPARATOR), filePath.length());
                }
                file = file.getParentFile();
            }
            return getWindowsShortPath(file.getAbsolutePath()) + lastPathAfterSeparator;
        }
    }

    private String getWindowsShortPath(String path) {
        if (path.length() >= 256) {
            char[] result = new char[256];

            //Call CKernel32 interface to execute GetShortPathNameA method
            Kernel32.INSTANCE.GetShortPathName(path, result, result.length);
            return Native.toString(result);
        }
        return path;
    }

    private boolean readBlock(InputStreamReader inputStreamReader, BufferedReader reader, ExecutorService executorService,
                              List<String> lines, boolean includeErrorLines) throws TimeoutException {
        boolean wasError = false;
        boolean continueReadingLines = true;
        boolean timeout = false;
        try {
            if (!includeErrorLines) {
                logger.debug("trying to read lines using '{}'", Arrays.toString(args));
            }
            int lineIndex = 1;
            String line = Constants.EMPTY_STRING;
            while (continueReadingLines && line != null) {
                Future<String> future = executorService.submit(new CommandLineProcess.ReadLineTask(reader));
                try {
                    line = future.get(this.timeoutReadLineSeconds, TimeUnit.SECONDS);
                    if (!includeErrorLines) {
                        if (StringUtils.isNotBlank(line)) {
                            logger.debug("Read line #{}: {}", lineIndex, line);
                            lines.add(line);
                        } else {
                            logger.debug("Finished reading {} lines", lineIndex - 1);
                        }
                    } else {
                        if (StringUtils.isNotBlank(line)) {
                            lines.add(line);
                        }
                    }
                } catch (TimeoutException e) {
                    logger.debug("Received timeout when reading line #" + lineIndex, e.getStackTrace());
                    logger.warn("Failed reading line, probably user input is expected.");
                    inputStreamReader = null;
                    reader = null;
                    continueReadingLines = false;
                    wasError = true;
                    timeout = true;
                } catch (Exception e) {
                    logger.debug("Error reading line #" + lineIndex, e.getStackTrace());
                    continueReadingLines = false;
                    wasError = true;
                }
                lineIndex++;
            }
        } catch (Exception e) {
            logger.error("error parsing output : {}", e.getStackTrace());
        } finally {
            executorService.shutdown();
            IOUtils.closeQuietly(inputStreamReader);
            IOUtils.closeQuietly(reader);
        }
        if (timeout) {
            throw new TimeoutException("Failed reading line, probably user input is expected.");
        } else {
            return wasError;
        }
    }

    public void executeProcessWithoutOutput() throws IOException {
        executeProcess(false, false);
    }

    public List<String> executeProcessWithErrorOutput() throws IOException {
        return executeProcess(false, true);
    }

    public void setTimeoutReadLineSeconds(long timeoutReadLineSeconds) {
        this.timeoutReadLineSeconds = timeoutReadLineSeconds;
    }

    public void setTimeoutProcessMinutes(long timeoutProcessMinutes) {
        this.timeoutProcessMinutes = timeoutProcessMinutes;
    }

    public boolean isErrorInProcess() {
        return this.errorInProcess;
    }

    public int getExitStatus() {
        if (processStart != null) {
            return processStart.exitValue();
        }
        return 0;
    }

    /* --- Nested classes --- */

    class ReadLineTask implements Callable<String> {

        /* --- Members --- */

        private final BufferedReader reader;

        /* --- Constructors --- */

        ReadLineTask(BufferedReader reader) {
            this.reader = reader;
        }

        /* --- Overridden methods --- */

        @Override
        public String call() throws Exception {
            return reader.readLine();
        }
    }
}
