package com.cloudbees.api.oauth;

import com.cloudbees.api.BeesClient;

import javax.annotation.CheckForNull;
import java.util.Collection;
import java.util.List;

/**
 * Defines OAuth-related CloudBees API.
 *
 * Also see {@link OauthClientServletUtil} that defines related convenience methods for webapps.
 *
 * @author Vivek Pandey
 * @see BeesClient#getOauthClient()
 */
public interface OauthClient {

    /**
     * Creates a OAuth token for the current user (used to create {@link BeesClient})
     *
     * @return OauthToken. always non-null if there was error such as invalid credentials
     * @throws OauthClientException if there is any error during token validation
     */
    public OauthToken createToken(TokenRequest tokenRequest) throws OauthClientException;

    /**
     * Deletes a token created by {@link #createToken(TokenRequest)}
     *
     * @see AbstractOauthToken#delete()
     */
    public void deleteToken(String oauthTokenId) throws OauthClientException;

    /**
     * Lists up all the tokens generated by te current user.
     */
    public List<OauthTokenDetail> listTokens() throws OauthClientException;

    /**
     * Validates token with the given scopes. Returns null if the given access token is invalid, otherwise OauthToken is returned.
     *
     * <p>
     * {@link BeesClient} must be constructed with OAuth client ID and client secret as the username and password.
     *
     * @param token non-null token
     * @param scopes array of scope that are expected to be granted for this token
     * @return null if the token is invalid such as expired or unknown to the CloudBees OAuth server or the expected
     * scopes are not found.
     */
    @CheckForNull
    public OauthToken validateToken(String token, String... scopes) throws OauthClientException;

    /**
     * Obtains the details of the token and performs minimal validation (such as expiration.)
     * Returns null if the given access token is invalid, otherwise OauthToken is returned.
     *
     * <p>
     * {@link BeesClient} must be constructed with OAuth client ID and client secret as the username and password.
     *
     * <h2>Caching</h2>
     * <p>
     * This method call is a network operation. If you are developing a server application that performs
     * frequent token validations, then consider using https://github.com/CloudBees-community/cached-token-validator ,
     * which adds caching on top of this.
     *
     * @param token non-null token
     * @return null if the token is invalid such as expired or unknown to the CloudBees OAuth server.
     */
    @CheckForNull
    public OauthToken validateToken(String token) throws OauthClientException;

    /**
     * Parses Bearer token from HTTP Authorization header
     *
     * @param authorizationHeader HTTP Authorization Header
     *
     * @return Returns null if there is no Bearer token found otherwise a String representing oauth token
     */
    public String parseAuthorizationHeader(String authorizationHeader);

    /**
     * Registers a new OAuth client application.
     *
     * @return
     *      A fully populated {@link OauthClientApplication} object. This is not the same object
     *      as you passed in the input.
     */
    public OauthClientApplication registerApplication(OauthClientApplication input) throws OauthClientException;

    /**
     * Gets the details of an OAuth client application by its client ID.
     *
     * @return never null. If the app isn't found, an exeption will be thrown.
     */
    public OauthClientApplication getApplication(String clientId) throws OauthClientException;

    /**
     * Lists up all the OAuth client applications registered by the current user.
     */
    public List<OauthClientApplication> listApplication() throws OauthClientException;

    void deleteApplication(String clientId) throws OauthClientException;

    /**
     * OAuth client application can use this method to exchange the authorization code
     * (which it gets from the browser after GC authenticates the user and redirects him back to you)
     * to the OAuth access token.
     *
     * <p>
     * For this method to work, {@link BeesClient} should be called with OAuth client ID and secret.
     *
     * @see <a href="http://wiki.cloudbees.com/bin/view/RUN/OAuth#HExchangeAuthorizationCodeForAccessToken">Wiki</a>
     *
     * @param redirectUri
     *      Required if present in the authorization request, and the value must be the same.
     * @return never null. In case of a problem, an exception will be thrown.
     */
    OauthToken exchangeToAccessToken(String authorizationCode, String redirectUri) throws OauthClientException;

    /**
     * Exchange refresh_token to an access token. The new access_token can be created with the same or subset of
     * original scopes the refresh token was granted for.
     *
     * @param refreshToken required. refresh_token.
     *
     * @param scopes optional. If not provided the returned access_token carries the same scopes as the one granted
     *               to refresh_token.
     * @return Valid OauthToken
     *
     * @throws OauthClientException
     */
    OauthToken exchangeToAccessToken(String refreshToken, String... scopes) throws OauthClientException;

    /**
     * OAuth client application can use this method to create an OAuth token with arbitrary scopes
     * that belongs to the user who registered the application.
     *
     * The created token will be tied only to the account that the OAuth client application is registered with,
     * even if the user who registered it may have access to other accounts.
     *
     * <p>
     * For this method to work, {@link BeesClient} should be called with OAuth client ID and secret.
     *
     * @see <a href="http://wiki.cloudbees.com/bin/view/RUN/OAuth#HServerApplication">Wiki</a>
     *
     * @return never null. In case of a problem, an exception will be thrown.
     */
    OauthToken createOAuthClientToken(Collection<String> scopes) throws OauthClientException;

    /**
     * Overloaded version of {@link #createOAuthClientToken(Collection)}
     */
    OauthToken createOAuthClientToken(String... scopes) throws OauthClientException;

    /**
     * Overloaded version of {@link #createOAuthClientToken(Collection)}
     *
     */
    OauthToken createOAuthClientToken(TokenRequest tokenRequest) throws OauthClientException;

}
