package com.cloudhopper.mq.broker;

/*
 * #%L
 * ch-mq
 * %%
 * Copyright (C) 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;

/**
 * Events (that can be delayed) occurring on a queue.
 * 
 * @author joelauer
 */
public class DistributedQueueEvent implements Delayed {

    // time this event should be run at
    private final long runTime;
    // name of queue event occurred on
    private final String queueName;
    // event triggered on this queue
    private final DistributedQueueManager.Event event;

    public DistributedQueueEvent(String queueName, DistributedQueueManager.Event event) {
        this(queueName, event, 0);
    }

    public DistributedQueueEvent(String queueName, DistributedQueueManager.Event event, long msToDelay) {
        this.queueName = queueName;
        this.event = event;
        if (msToDelay > 0) {
            this.runTime = System.currentTimeMillis() + msToDelay;
        } else {
            this.runTime = 0;
        }
    }

    public String getQueueName() {
        return this.queueName;
    }

    public DistributedQueueManager.Event getEvent() {
        return event;
    }

    public long getDelay(TimeUnit unit) {
        // if msToDelay is zero, that means run it immediately
        if (runTime <= 0) {
            return 0;
        }
        // calculate number of milliseconds remaining
        long remainingTime = runTime - System.currentTimeMillis();
        // convert time into unit requested and return it
        return unit.convert(remainingTime, TimeUnit.MILLISECONDS);
    }

    public int compareTo(Delayed o) {
        long myDelay = this.getDelay(TimeUnit.MILLISECONDS);
        long otherDelay = o.getDelay(TimeUnit.MILLISECONDS);
        if (myDelay < otherDelay) {
            return -1;
        } else if (myDelay > otherDelay) {
            return 1;
        } else {
            return 0;
        }
    }

    @Override
    public String toString() {
        return new StringBuilder(40)
            .append("[queueName=")
            .append(queueName)
            .append(", event=")
            .append(event)
            .append("]")
            .toString();
    }

}
