package com.cloudhopper.mq.broker;

/*
 * #%L
 * ch-mq-remote
 * %%
 * Copyright (C) 2009 - 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The shared view of brokers and queues replicated across the entire group
 * of brokers.  Used by the Distributor to figure out where & how to try and
 * forward items from a queue.
 *
 *   Broker1
 *      Queue0 - 2
 *      Queue1 - 1
 *   Broker2
 *      Queue0 - 1
 *      Queue1 - 2
 *
 *   Queue0
 *      Broker1 - 2
 *      Broker2 - 1
 *   Queue1
 *      Broker1 - 1
 *      Broker2 - 2
 *
 * @author joelauer
 */
public class DistributedQueueState {
    private static final Logger logger = LoggerFactory.getLogger(DistributedQueueState.class);

    // configuration of the distributed queue
    private final DistributedQueueConfiguration configuration;
    // listeners watching for state changes
    private final CopyOnWriteArrayList<DistributedQueueStateListener> listeners;
    // map of urls to remote brokers
    private final ConcurrentHashMap<String,RemoteBrokerInfo> remoteBrokers;
    // map of names to remote queues
    private final ConcurrentHashMap<String,RemoteQueueInfo> remoteQueues;

    public DistributedQueueState(DistributedQueueConfiguration configuration) {
        this.configuration = configuration;
        this.listeners = new CopyOnWriteArrayList<DistributedQueueStateListener>();
        this.remoteBrokers = new ConcurrentHashMap<String,RemoteBrokerInfo>();
        this.remoteQueues = new ConcurrentHashMap<String,RemoteQueueInfo>();

    }

    /**
     * Clears all internal structures containing state information about
     * remote brokers or remote queues.  Does not remove listeners.
     */
    public void clear() {
        for (RemoteBrokerInfo bi : this.remoteBrokers.values()) {
            bi.clear();
        }
        this.remoteBrokers.clear();

        for (RemoteQueueInfo qi : this.remoteQueues.values()) {
            qi.clear();
        }
        this.remoteQueues.clear();
    }

    public void addListener(DistributedQueueStateListener listener) {
        this.listeners.addIfAbsent(listener);
    }

    public void removeListener(DistributedQueueStateListener listener) {
        this.listeners.remove(listener);
    }

    public RemoteBrokerInfo getRemoteBroker(String url) {
        return this.remoteBrokers.get(url);
    }

    public ConcurrentHashMap<String,RemoteBrokerInfo> getRemoteBrokers() {
        return this.remoteBrokers;
    }

    public void addRemoteBroker(RemoteBrokerInfo bi) {
        // add the broker if it didn't already exist
        RemoteBrokerInfo oldbi = this.remoteBrokers.putIfAbsent(bi.getUrl(), bi);

        // was the item added?
        if (oldbi == null) {
            // propagate event to all listeners
            for (DistributedQueueStateListener listener : listeners) {
                try {
                    listener.notifyRemoteBrokerAdded(bi);
                } catch (Throwable t) {
                    logger.error("Unable to cleanly propagate RemoteBroker @ " + bi.getUrl() + " was added", t);
                }
            }
        }
    }

    public void removeRemoteBroker(String url) {
        // atomically remove the key
        RemoteBrokerInfo oldbi = this.remoteBrokers.remove(url);

        // was it removed?
        if (oldbi != null) {

            // FIXME: remove the RemoteBroker for ANY queues it was in???

            // propagate event to all listeners
            for (DistributedQueueStateListener listener : listeners) {
                try {
                    listener.notifyRemoteBrokerRemoved(oldbi);
                } catch (Throwable t) {
                    logger.error("Unable to cleanly propagate RemoteBroker was removed", t);
                }
            }
        }
    }

    public RemoteQueueInfo getRemoteQueue(String queueName) {
        return this.remoteQueues.get(queueName);
    }

    public ConcurrentHashMap<String,RemoteQueueInfo> getRemoteQueues() {
        return this.remoteQueues;
    }

    /**
     * Indicates that an item was sent to a remote broker and remote queue.
     * @param url
     * @param queueName
     * @param itemCount
     */
    /**
    public void sentItemsToRemoteQueue(String url, String queueName, int itemCount) {
        // get a remote broker
        RemoteBrokerInfo bi = this.getRemoteBroker(url);
        if (bi != null) {
            bi.incrementSent(itemCount);
        }
    }
     */

    /**
     * Updates the state of a RemoteBroker.  If the state actually changed,
     * this method will trigger a notifyRemoteBrokerStateChanged event to
     * any listener.
     * @param url The url of the RemoteBroker
     * @param state The new state of the RemoteBroker
     * @param errorMessage The new error message to save with the RemoteBroker
     * @param monitorTime The timestamp of the monitoring event that triggered
     *      this state change.
     */
    synchronized public void updateRemoteBrokerState(String url, int state, String errorMessage, long monitorTime) {
        // get a remote broker
        RemoteBrokerInfo bi = this.getRemoteBroker(url);

        // did it exist?
        if (bi == null) {
            logger.warn("Unable to update state of RemoteBroker @ [" + url + "] since its not in our master list");
            return;
        }

        // set the state and see if the state actually changed
        boolean changed = bi.setState(state);

        // only update the monitor time if it was OK
        if (state == RemoteBrokerInfo.STATE_AVAILABLE) {
            // update last available time to the current monitor time
            bi.setLastAvailableTime(monitorTime);
        } else {
            // we need to remove this broker from every queue
            String[] queueNames = bi.getRemoteQueues().toArray(new String[0]);
            for (String queueName : queueNames) {
                this.updateRemoteQueueWeight(queueName, bi.getUrl(), 0);
            }
        }

        // always update the error message
        bi.setLastErrorMessage(errorMessage);

        // if the state changed, propagate the new state
        if (changed) {
            for (DistributedQueueStateListener listener : listeners) {
                try {
                    listener.notifyRemoteBrokerStateChanged(bi, state);
                } catch (Throwable t) {
                    logger.error("Unable to cleanly propagate RemoteBroker state change notification", t);
                }
            }
        }
    }


    synchronized public void updateRemoteQueueWeight(String queueName, String url, int weight) {
        // a remote broker MUST exist prior to being able to update a queue
        RemoteBrokerInfo bi = this.getRemoteBroker(url);

        if (bi == null) {
            logger.warn("Unable to update weight of RemoteQueue [" + queueName + "] since there is no associated RemoteBroker @ [" + url + "] in our master list");
            return;
        }

        // for state changes below
        boolean queueAdded = false;

        // 99% of the time, a RemoteQueue will already exist
        RemoteQueueInfo qi = this.remoteQueues.get(queueName);

        // weight of zero means remove it
        if (weight <= 0) {
            
            // if the remote queue wasn't found
            if (qi == null) {
                // this isn't really an error and is a very likely response
                logger.debug("Unable to update weight of RemoteQueue [" + queueName + "] since it doesn't exist yet AND the weight was zero");
                return;
            }

            // make sure the queue name is removed from the remote broker
            bi.removeRemoteQueue(queueName);
            
        // set the new weight for the broker and queue
        } else {
            // if remote queue was null, we need to add it
            if (qi == null) {
                logger.debug("RemoteQueue " + queueName + " did not exist, adding it");

                // create a new RemoteQueue based on the local distributed queue config
                qi = new RemoteQueueInfo(queueName, configuration.getAreaId());

                RemoteQueueInfo oldqi = this.remoteQueues.putIfAbsent(queueName, qi);

                // if qi == null, then the calling thread won on putting it
                if (oldqi == null) {
                    queueAdded = true;
                } else {
                    // use the entry in the remoteQueues
                    qi = oldqi;
                }
            }

            // make sure the queue name is part of the broker
            bi.addRemoteQueue(queueName);
        }

        // set or add the remote broker and its weight
        RemoteQueueInfo.Event weightEvent = qi.setRemoteBrokerWeight(bi, weight);

        if (queueAdded) {
            for (DistributedQueueStateListener listener : listeners) {
                try {
                    listener.notifyRemoteQueueAdded(qi);
                } catch (Throwable t) {
                    logger.error("Unable to cleanly propagate RemoteQueue was added notification", t);
                }
            }
        }

        // so either there was a state change (highest precedence) or it was
        // an attributes change (lower precedence) or nothing changed
        if (weightEvent == RemoteQueueInfo.Event.STATE_CHANGE) {
            for (DistributedQueueStateListener listener : listeners) {
                try {
                    listener.notifyRemoteQueueStateChanged(qi, qi.getState());
                } catch (Throwable t) {
                    logger.error("Unable to cleanly propagate RemoteQueue state changed notification", t);
                }
            }
        } else if (weightEvent == RemoteQueueInfo.Event.ATTRIBUTES_CHANGE) {
            for (DistributedQueueStateListener listener : listeners) {
                try {
                    listener.notifyRemoteQueueAttributesChanged(qi);
                } catch (Throwable t) {
                    logger.error("Unable to cleanly propagate RemoteQueue attributes changed notification", t);
                }
            }
        } 
    }

    public String toDebugString() {
        StringBuilder buf = new StringBuilder(200);
        
        buf.append("\nDistributedQueueState\n");

        for (RemoteBrokerInfo bi : this.remoteBrokers.values()) {
            buf.append(" RemoteBroker ");
            buf.append(bi.toString());
            buf.append("\n");

            // append every entry for the remote queue
            for (String queueName : bi.getRemoteQueues()) {
                buf.append(" -> RemoteQueue [" + queueName + "]\n");
            }
        }

        for (RemoteQueueInfo qi : this.remoteQueues.values()) {
            buf.append(" RemoteQueue ");
            buf.append(qi.toString());
            buf.append("\n");

            // append every entry for the remote queue
            for (Map.Entry<String,RemoteQueueInfo.Entry> entry : qi.getRemoteBrokers().entrySet()) {
                buf.append(" -> RemoteBroker [" + entry.getKey() + "] [weight=" + entry.getValue().weight.get() + ", sent=" + entry.getValue().sent.get() + "]\n");
            }

        }

        return buf.toString();
    }

}
