package com.cloudhopper.mq.broker;

/*
 * #%L
 * ch-mq-remote
 * %%
 * Copyright (C) 2009 - 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.Condition;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An aggressively locked, no dupes allowed, BlockingQueue. Used in FairRemoteQueueTransferScheduler only.
 *
 * @author garth
 */
public class NoDupBlockingQueue<E> {
    private static final Logger logger = LoggerFactory.getLogger(NoDupBlockingQueue.class);

    private final Set<E> set;
    private final ReentrantLock lock;
    private final Condition notEmpty;
    private final LinkedList<E> list;
    private int count;

    public NoDupBlockingQueue() {
	this.set = new HashSet<E>();
	this.list = new LinkedList<E>();
	this.count = 0;
        this.lock = new ReentrantLock(true);
	this.notEmpty = lock.newCondition();
    }

    public E take() throws InterruptedException {
	lock.lockInterruptibly();
	try {
	    while (count == 0) notEmpty.await();
	    E item = list.poll();
	    set.remove(item);
	    --count;
	    return item;
	} finally {
	    lock.unlock();
	}
    }

    public boolean remove(Object o) {
	lock.lock();
	try {
	    if (set.remove(o)) {
		list.remove(o);
		--count;
		return true;
	    } else {
		return false;
	    }
	} finally {
	    lock.unlock();
	}
    }

    public boolean add(E e) {
	lock.lock();
	try {
	    if (set.add(e)) {
		list.offer(e);
		++count;
		notEmpty.signal();
		return true;
	    } else {
		return false;
	    }
	} finally {
	    lock.unlock();
	}
    }

    public int drainTo(Collection<? super E> c) {
        if (c == null)
            throw new NullPointerException();
        if (c == this)
            throw new IllegalArgumentException();
	lock.lock();
	try {
            int n = 0;
            int max = count;
            while (n < max) {
		E item = list.poll();
		c.add(item);
		set.remove(item);
		++n;
	    }
	    if (n > 0) {
		count = 0;
	    }
            return n;
	} finally {
	    lock.unlock();
	}
    }

    public int size() {
	lock.lock();
	try {
	    int setSize = set.size();
	    int queueSize = list.size();
	    if (setSize != queueSize) {
		logger.error("{} != {}. THIS SHOULD NEVER HAPPEN.", setSize, queueSize);
	    }
	    return count; 
	} finally {
	    lock.unlock();
	}
    }

}
