package com.cloudhopper.mq.broker;

/*
 * #%L
 * ch-mq
 * %%
 * Copyright (C) 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.cloudhopper.mq.broker.protocol.ProtocolConstants;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.joda.time.DateTime;

/**
 * Immutable information about a remote broker.
 * 
 * @author joelauer
 */
public class RemoteBrokerInfo implements Comparable<RemoteBrokerInfo> {
    private static final Logger logger = LoggerFactory.getLogger(RemoteBrokerInfo.class);

    /** all possible states of a RemoteBroker */
    public final static int STATE_NOT_AVAILABLE = 0;
    public final static int STATE_AVAILABLE = 1;
    
    public final static String[] STATES = new String[] {
        "Not Available", "Available"
    };

    // url of remote broker
    private final String url;
    // id of remote broker
    private final AtomicInteger id;
    // area of broker (for multiple data centers)
    private final AtomicInteger areaId;
    // last time the broker was monitored (only updated by a RemoteBrokerMonitor)
    private final AtomicLong lastMonitorTime;
    // state of remote broker
    private final AtomicInteger state;
    // number of items sent
    private final AtomicLong sent;
    // last time the state changed
    private final AtomicLong lastStateChangedTime;
    // last time the broker had its state updated to AVAILABLE (even if it was available before)
    private final AtomicLong lastAvailableTime;
    // last message about the state
    private String lastErrorMessage;
    // version of protocol supported by remote broker
    private byte version;
    // every queue we can reach
    private final Set<String> remoteQueues;

    public RemoteBrokerInfo(String url) {
        this.url = url;
        this.id = new AtomicInteger();
        this.areaId = new AtomicInteger();
        this.lastStateChangedTime = new AtomicLong(System.currentTimeMillis());
        this.lastMonitorTime = new AtomicLong(0);
        this.lastAvailableTime = new AtomicLong(0);
        this.state = new AtomicInteger(STATE_NOT_AVAILABLE);
        this.lastErrorMessage = "Not yet monitored";
        this.sent = new AtomicLong(0);
	this.version = ProtocolConstants.VERSION_1_0;
        this.remoteQueues = Collections.synchronizedSet(new HashSet<String>());
    }

    /**
     * Clears all internal data structures such as remote brokers and any
     * load balancer lists.
     */
    public void clear() {
        this.remoteQueues.clear();
    }

    public String getUrl() {
        return this.url;
    }

    /**
     * Returns a set of all RemoteQueues by name that are mapped to this
     * RemoteBroker.
     * @return A Set of all remote queue names mapped to this broker
     */
    public Set<String> getRemoteQueues() {
        return this.remoteQueues;
    }

    public boolean addRemoteQueue(String queueName) {
        return this.remoteQueues.add(queueName);
    }

    public boolean removeRemoteQueue(String queueName) {
        return this.remoteQueues.remove(queueName);
    }

    public void clearRemoteQueues() {
        this.remoteQueues.clear();
    }

    public int getId() {
        return this.id.get();
    }

    protected void setId(int value) {
        this.id.set(value);
    }

    public int getAreaId() {
        return this.areaId.get();
    }

    protected void setAreaId(int value) {
        this.areaId.set(value);
    }

    public long getLastMonitorTime() {
        return this.lastMonitorTime.get();
    }

    protected void setLastMonitorTime(long value) {
        this.lastMonitorTime.set(value);
    }

    /**
     * Gets the last time this had its state set to available (even if it was
     * previously available).  Let's you track that an update was made even if
     * it didn't change the actual state.
     * @return The last time this had its state set to available
     */
    public long getLastAvailableTime() {
        return this.lastAvailableTime.get();
    }

    protected void setLastAvailableTime(long value) {
        this.lastAvailableTime.set(value);
    }

    public int getState() {
        return this.state.get();
    }

    public void incrementSent(int count) {
        this.sent.addAndGet(count);
    }

    public long getSent() {
        return this.sent.get();
    }

    /**
     * Sets the state of the remote broker to the new value.  If the state was
     * changed, it returns true.  If the state is the same, returns false.
     * @param value
     * @return true if changed
     */
    protected boolean setState(int value) {
        int oldValue = this.state.getAndSet(value);
        if (oldValue != value) {
            this.lastStateChangedTime.set(System.currentTimeMillis());
            logger.info("RemoteBroker @ " + url + " changed from " + STATES[oldValue] + " to " + STATES[value]);
            return true;
        } else {
            return false;
        }
    }

    public long getLastStateChangedTime() {
        return this.lastStateChangedTime.get();
    }

    public void setLastErrorMessage(String value) {
        this.lastErrorMessage = value;
    }

    public String getLastErrorMessage() {
        return this.lastErrorMessage;
    }

    public void setVersion(byte value) {
	this.version = value;
    }

    public byte getVersion() {
	return this.version;
    }

    /**
     * Checks whether this remote broker is available and ready to have items
     * from the queue forwarded to it.
     * @return true if the broker is available
     */
    public boolean isAvailable() {
        return (this.state.get() == STATE_AVAILABLE);
    }

    /**
     * Checks whether this remote broker is not available.
     * @return true if the broker isn't available
     */
    public boolean isNotAvailable() {
        return (this.state.get() != STATE_AVAILABLE);
    }

    @Override
    public boolean equals(Object otherObject) {
        if (otherObject == null) {
            return false;
        }
        if (!(otherObject instanceof RemoteBrokerInfo)) {
            return false;
        }
        RemoteBrokerInfo info = (RemoteBrokerInfo)otherObject;
        return (this.url.equals(info.url));
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 29 * hash + (this.url != null ? this.url.hashCode() : 0);
        return hash;
    }

    public int compareTo(RemoteBrokerInfo o) {
        // always compare to the other broker by uuid
        return this.url.compareTo(o.url);
    }

    @Override
    public String toString() {
        return new StringBuilder(50)
            .append("[url=")
            .append(getUrl())
            .append(", state=")
            .append(STATES[getState()])
            .append(", areaId=")
            .append(getAreaId())
            .append(", lastMonitorTime=")
            .append(new DateTime(getLastMonitorTime()))
            .append("}")
            .toString();
    }
}
