package com.cloudhopper.mq.broker;

/*
 * #%L
 * ch-mq
 * %%
 * Copyright (C) 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.cloudhopper.commons.util.LoadBalancedList;
import com.cloudhopper.commons.util.LoadBalancedLists;
import com.cloudhopper.commons.util.RoundRobinLoadBalancedList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Immutable information about a remote queue on one or more remote brokers...
 * 
 * @author joelauer
 */
public class RemoteQueueInfo implements Comparable<RemoteQueueInfo> {
    private static final Logger logger = LoggerFactory.getLogger(RemoteQueueInfo.class);

    /** all possible states of a RemoteQueue */
    public final static int STATE_NOT_AVAILABLE = 0;
    public final static int STATE_AVAILABLE = 1;
    
    public final static String[] STATES = new String[] {
        "Not Available", "Available"
    };

    // what we track for each remote broker tied to a queue
    public static class Entry {
        public final AtomicInteger weight;
        public final AtomicLong sent;
	public final RemoteBrokerInfo broker;
        public Entry(RemoteBrokerInfo broker) {
            this(broker, 0);
        }
         public Entry(RemoteBrokerInfo broker, int weight) {
	    this.broker = broker;
            this.weight = new AtomicInteger(weight);
            this.sent = new AtomicLong();
        }
	public RemoteBrokerInfo getBroker() {
	    return this.broker;
	} 
   }

    // enumeration of events that setting the weight could return
    public enum Event {
        NO_CHANGE,
        STATE_CHANGE,
        ATTRIBUTES_CHANGE
    };

    // name of queue
    private final String name;
    // areaId of local broker
    private final int localAreaId;
    // state of remote queue
    private final AtomicInteger state;
    // map of remote brokers and their current weight and counters
    private final ConcurrentHashMap<String,Entry> remoteBrokers;
    // round robin load balanced list (primary: for "our" area)
    private final LoadBalancedList<String> primaryLoadBalancer;
    // round robin load balanced list (failover: for "other" areas)
    private final LoadBalancedList<String> failoverLoadBalancer;
    // last time the state changed
    private final AtomicLong lastStateChangedTime;

    public RemoteQueueInfo(String name, int localAreaId) {
        this.name = name;
        this.localAreaId = localAreaId;
        this.state = new AtomicInteger(STATE_NOT_AVAILABLE);
        this.remoteBrokers = new ConcurrentHashMap<String,Entry>();
        this.primaryLoadBalancer = LoadBalancedLists.synchronizedList(new RoundRobinLoadBalancedList<String>());
        this.failoverLoadBalancer = LoadBalancedLists.synchronizedList(new RoundRobinLoadBalancedList<String>());
        this.lastStateChangedTime = new AtomicLong(System.currentTimeMillis());
    }

    /**
     * Clears all internal data structures such as remote brokers and any
     * load balancer lists.
     */
    public void clear() {
        this.failoverLoadBalancer.clear();
        this.primaryLoadBalancer.clear();
        this.remoteBrokers.clear();
    }

    public String getName() {
        return this.name;
    }

    public int getLocalAreaId() {
        return this.localAreaId;
    }

    public ConcurrentHashMap<String,Entry> getRemoteBrokers() {
        return this.remoteBrokers;
    }

    /**
     * Gets the total number of RemoteBrokers (either primary or failover) that
     * can consume this queue.
     * @return The total number of RemoteBrokers that can consume this queue
     */
    public int getSize() {
        return this.remoteBrokers.size();
    }

    /**
     * Gets the total number of primary RemoteBrokers that can consume this queue.
     * A primary RemoteBroker is a broker that is in the local brokers area.
     * @return The total number of primary RemoteBrokers that can consume this queue
     */
    public int getPrimarySize() {
        return this.primaryLoadBalancer.getSize();
    }

    public List<LoadBalancedList.Node<String>> getPrimaryList() {
        return this.primaryLoadBalancer.getValues();
    }

    /**
     * Gets the total number of failover RemoteBrokers that can consume this queue.
     * A failover RemoteBroker is a broker that isn't in the local brokers area.
     * @return The total number of failover RemoteBrokers that can consume this queue
     */
    public int getFailoverSize() {
        return this.failoverLoadBalancer.getSize();
    }

    public List<LoadBalancedList.Node<String>> getFailoverList() {
        return this.failoverLoadBalancer.getValues();
    }

    /**
     * Gets the URL for the next RemoteBroker to send a request to.  Internally
     * uses the primary and failover load balancer lists.
     * @return The next RemoteBroker URL to send to or null if no more RemoteBrokers
     *      are available.
     */
    public String getNextRemoteBroker() {
        String url = this.primaryLoadBalancer.getNext();
        if (url == null) {
            return this.failoverLoadBalancer.getNext();
        } else {
            return url;
        }
    }

    /**
     * Sets or adds the RemoteBroker and either updates or sets the weight
     * of it to the new value.
     * @param bi The RemoteBroker we're setting the weight for
     * @param weight The new or updated weight of the broker. Setting to zero
     *      removes the RemoteBroker from our internal structures.
     * @return True if the weight caused a state change, otherwise false.
     */
    public Event setRemoteBrokerWeight(RemoteBrokerInfo bi, int weight) {
        // 90% of the time the entry will already exist
        Entry entry = this.remoteBrokers.get(bi.getUrl());

        // assume attributes didn't change
        boolean attributesChanged = false;

        if (entry == null) {
            // entry did not exist -- what do we do?
            if (weight <= 0) {
                // nothing to do
                return Event.NO_CHANGE;
            }
            
            // create a new entry
            entry = new Entry(bi);
            
            // set/add the entry if it didn't already exist
            Entry oldentry = this.remoteBrokers.putIfAbsent(bi.getUrl(), entry);
            if (oldentry != null) {
                entry = oldentry;
            }
        }

        if (weight <= 0) {
            // remove the RemoteBroker from everything
            Entry curEntry = this.remoteBrokers.remove(bi.getUrl());

            // if there was an entry, the attributes at least changed
            if (curEntry != null) {
                attributesChanged = true;
            }

            // decide which load balanced list to remove it from
            if (localAreaId == bi.getAreaId()) {
                this.primaryLoadBalancer.remove(bi.getUrl());
            } else {
                this.failoverLoadBalancer.remove(bi.getUrl());
            }
        } else {
            // set the weight
            int oldWeight = entry.weight.getAndSet(weight);

            // if the weight changed, make sure to track it
            if (oldWeight != weight) {
                attributesChanged = true;
            }

            // if the RemoteBroker areaId matches our localAreaId, add this to our
            // primary load balancer list
            if (localAreaId == bi.getAreaId()) {
                this.primaryLoadBalancer.set(bi.getUrl(), weight);
            } else {
                this.failoverLoadBalancer.set(bi.getUrl(), weight);
            }
        }        

        // the RemoteQueue is "available" if the size of either load balancer list is > 0
        //int size = this.primaryLoadBalancer.getSize() + this.failoverLoadBalancer.getSize();
        int size = getSize();

        // default stateChanged to false
        boolean stateChanged = false;

        // see if we should update the state
        if (size <= 0) {
            stateChanged = setState(STATE_NOT_AVAILABLE);
        } else {
            stateChanged = setState(STATE_AVAILABLE);
        }

        // k, so if we had a state change, return that since it takes precedence
        if (stateChanged) {
            return Event.STATE_CHANGE;
        } else if (attributesChanged) {
            return Event.ATTRIBUTES_CHANGE;
        } else {
            return Event.NO_CHANGE;
        }
    }

    public int getState() {
        return this.state.get();
    }

    protected boolean setState(int value) {
        int oldValue = this.state.getAndSet(value);
        if (oldValue != value) {
            this.lastStateChangedTime.set(System.currentTimeMillis());
            logger.debug("RemoteQueue " + name + " changed state from " + STATES[oldValue] + " to " + STATES[value]);
            return true;
        } else {
            return false;
        }
    }

    public long getLastStateChangedTime() {
        return this.lastStateChangedTime.get();
    }

    public boolean isAvailable() {
        return (this.state.get() == STATE_AVAILABLE);
    }

    public boolean isNotAvailable() {
        return (this.state.get() != STATE_AVAILABLE);
    }

    @Override
    public boolean equals(Object otherObject) {
        if (otherObject == null) {
            return false;
        }
        if (!(otherObject instanceof RemoteQueueInfo)) {
            return false;
        }
        RemoteQueueInfo info = (RemoteQueueInfo)otherObject;
        return (this.name.equals(info.name));
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 29 * hash + (this.name != null ? this.name.hashCode() : 0);
        return hash;
    }

    public int compareTo(RemoteQueueInfo o) {
        // always compare to the other broker by uuid
        return this.name.compareTo(o.name);
    }

    @Override
    public String toString() {
        return new StringBuilder(50)
            .append("[name=")
            .append(getName())
            .append(", state=")
            .append(STATES[getState()])
            .append(", brokerSize=")
            .append(getSize())
            .append(", primaryBrokerSize=")
            .append(getPrimarySize())
            .append(", failoverBrokerSize=")
            .append(getFailoverSize())
            .append("]")
            .toString();
    }

}
