package com.cloudhopper.mq.broker;

/*
 * #%L
 * ch-mq
 * %%
 * Copyright (C) 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

/**
 * Interface to handle notifications during a transfer that a RemoteBroker
 * should be removed. This is usually useful when multiple RemoteBrokers are
 * available on a RemoteQueue and we should remove them from the list.
 *
 * @author joelauer
 */
public interface RemoteQueueTransferListener {

    /**
     * Callback when an entire remote broker is not available.  This is only
     * triggered when something like an IOException occurs and that means that
     * likely every RemoteQueue is now no longer available on this system.
     *
     * NOTE: The action to take when this callback occurs is to call
     * updateRemoteBrokerState with a state of NOT_AVAILABLE on the DistributedQueueState.
     *
     * @param remoteBrokerUrl The url of the RemoteBroker to mark as Not Available
     */
    public void notifyRemoteBrokerIsNoLongerAvailable(String remoteBrokerUrl, String errorMessage);

    /**
     * Callback when only a specific queue is now no longer available on a Remote
     * Broker.  This is triggered when something like a transfer response of
     * "No Local Queue" is received from a Remote Broker.  This error indicates
     * that the Remote Broker is still "up", but that just this particular
     * queue is now no longer available.
     *
     * NOTE: The action to take when this callback occurs is to call
     * updateRemoteQueueWeight with a "zero weight" on the DistributedQueueState.
     *
     * @param remoteBrokerUrl The url of the RemoteBroker to mark as Not Available
     * @param queueName The name of the queue
     */
    public void notifyQueueOnRemoteBrokerIsNoLongerAvailable(String remoteBrokerUrl, String queueName);

}
