package com.cloudhopper.mq.broker;

/*
 * #%L
 * ch-mq-remote
 * %%
 * Copyright (C) 2009 - 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.cloudhopper.mq.queue.Queue;
import com.cloudhopper.mq.queue.QueueListener;
import com.cloudhopper.mq.queue.QueueManager;
import java.util.AbstractQueue;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.management.ObjectName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Abstract scheduler implementing QueueListener and BlockingQueue. Implementations of scheduler
 * must implement onNotEmpty of QueueListener and take, drainTo, and size of BlockingQueue.
 * 
 * @author garth
 */
public abstract class RemoteQueueTransferScheduler
    extends AbstractQueue<Runnable>
    implements BlockingQueue<Runnable>, QueueListener, RemoteQueueTransferSchedulerMBean 
{
    private static final Logger logger = LoggerFactory.getLogger(RemoteQueueTransferScheduler.class);

    protected final Map<Queue, LocalToRemoteQueueProcessor> processors;
    protected final AtomicLong scheduledCount;
    protected int maxConcurrentRequests;

    public RemoteQueueTransferScheduler(int maxConcurrentRequests, QueueManager queueManager) {
	this.maxConcurrentRequests = maxConcurrentRequests;
	this.processors = Collections.synchronizedMap(new HashMap<Queue, LocalToRemoteQueueProcessor>());
	this.scheduledCount = new AtomicLong();
        registerMBean(queueManager);
    }

    protected void registerMBean(QueueManager queueManager) {
        if (queueManager.getConfiguration().isJmxEnabled()) {
            // register the this queue manager as an mbean
            try {
                ObjectName name = new ObjectName(queueManager.getConfiguration().getJmxDomain() + ":name=RemoteQueueTransferScheduler");
                queueManager.getConfiguration().getMBeanServer().registerMBean(this, name);
            } catch (Exception e) {
                // log the error, but don't throw an exception for this datasource
                logger.warn("Error while attempting to register DistributedQueueManager as an MBean: {}", e.toString());
            }
        }
    }

    public void addLocalToRemoteQueueProcessor(Queue queue, LocalToRemoteQueueProcessor processor) {
	logger.debug("[{}] Adding LocalToRemoteQueueProcessor for Queue to scheduler.", queue.getName());
	processors.put(queue, processor);
	queue.setQueueListener(this);
    }

    public void removeLocalToRemoteQueueProcessor(Queue queue) {
	logger.debug("[{}] Removing LocalToRemoteQueueProcessor for Queue from scheduler.", queue.getName());
	queue.setQueueListener(null);
	processors.remove(queue);
    }

    public int getMaxConcurrentRequests() {
	return this.maxConcurrentRequests;
    }

    public void setMaxConcurrentRequests(int maxConcurrentRequests) {
	this.maxConcurrentRequests = maxConcurrentRequests;
    }

    public int getLocalToRemoteQueueProcessorSize() {
	return processors.size();
    }

    public int getWorkListSize() {
	return size();
    }

    public long getScheduledCount() {
	return scheduledCount.get();
    }

    // QueueListener

    public abstract void onNotEmpty(Queue queue);

    // BlockingQueue. Need to support take(), drainTo() and size() for a ThreadPoolExecutor.

    public abstract Runnable take() throws InterruptedException; 

    public abstract int drainTo(Collection<? super Runnable> c);

    public abstract int size();

    // BlockingQueue. Everything else throws an UnsupportedOperationException.

    public boolean add(Runnable runnable) {
	logger.trace("add(Runnable)");
	throw new UnsupportedOperationException();
    }

    public boolean contains(Object o) {
	logger.trace("contains(Object)");
	throw new UnsupportedOperationException();
    }

    public int drainTo(Collection<? super Runnable> c, int maxElements) {
	logger.trace("drainTo(Collection, int)");
	throw new UnsupportedOperationException();
    }

    public boolean offer(Runnable runnable) {
	logger.trace("offer(Runnable)");
	throw new UnsupportedOperationException();
    }

    public boolean offer(Runnable runnable, long timeout, TimeUnit unit) {
	logger.trace("offer(Runnable, long, TimeUnit)");
	throw new UnsupportedOperationException();
    }

    public Runnable peek() {
	logger.trace("peek()");
	throw new UnsupportedOperationException();
    }

    public Runnable poll() {
	logger.trace("poll()");
	throw new UnsupportedOperationException();
    }

    public Runnable poll(long timeout, TimeUnit unit) {
	logger.trace("poll(long, TimeUnit)");
	throw new UnsupportedOperationException();
    }

    public void put(Runnable runnable) {
	logger.trace("put(Runnable)");
	throw new UnsupportedOperationException();
    }

    public int remainingCapacity() {
	logger.trace("remainingCapacity()");
	return Integer.MAX_VALUE;
    }

    public boolean remove(Object o) {
	logger.trace("remove(Object)");
	throw new UnsupportedOperationException();
    }

    public Iterator<Runnable> iterator() {
	logger.trace("iterator()");
	//	throw new UnsupportedOperationException();
	List<Runnable> l = new ArrayList<Runnable>();
	drainTo(l);
	return l.iterator();
    }

}
