package com.cloudhopper.mq.broker.protocol;

/*
 * #%L
 * ch-mq
 * %%
 * Copyright (C) 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.cloudhopper.commons.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Parses the DistributedQueue protocol.
 * 
 * @author joelauer
 */
public class ProtocolFactory {
    private static final Logger logger = LoggerFactory.getLogger(ProtocolFactory.class);

    static public String[] splitBody(String body) throws ProtocolParsingException {
        if (StringUtil.isEmpty(body)) {
            throw new ProtocolParsingException("Unable to split body into lines as body was empty");
        }

        return body.split("\n");
    }

    static public String[] splitKeyValue(String line) throws ProtocolParsingException {
        int pos = line.indexOf(':');
        if (pos <= 0) {
            throw new ProtocolParsingException("Unable to split line [" + line + "] into a key and value");
        }

        String[] kv = new String[2];

        kv[0] = line.substring(0, pos);

        // is there a value?
        if (pos+1 >= line.length()) {
            throw new ProtocolParsingException("The line [" + line + "] resulted in an empty value");
        }

        kv[1] = line.substring(pos+1);

        return kv;
    }

    static public byte parseVersionValue(String value) throws ProtocolParsingException {
        try {
            return Byte.parseByte(value);
        } catch (Exception e) {
            throw new ProtocolParsingException("Unable to parse [" + value + "] into version byte", e);
        }
    }

    static public int parseResultCodeValue(String value) throws ProtocolParsingException {
        try {
            return Integer.parseInt(value);
        } catch (Exception e) {
            throw new ProtocolParsingException("Unable to parse [" + value + "] into resultCode integer", e);
        }
    }

    static public int parseAreaIdValue(String value) throws ProtocolParsingException {
        try {
            return Integer.parseInt(value);
        } catch (Exception e) {
            throw new ProtocolParsingException("Unable to parse [" + value + "] into areaId integer", e);
        }
    }

    static public void processQueueValue(String value, MonitorResponse response) throws ProtocolParsingException {
        String[] v = value.split(",");

        if (v.length != 2) {
            throw new ProtocolParsingException("Unable to parse [" + value + "] into a queue name and weight - not 2 tokens");
        }

        int weight = 0;
        try {
            weight = Integer.parseInt(v[1]);
        } catch (Exception e) {
            throw new ProtocolParsingException("Unable to parse [" + v[1] + "] into a queue weight integer", e);
        }

        response.addQueue(v[0], weight);
    }

    static public boolean handleBaseResponseKeys(BaseResponse response, String key, String value) throws ProtocolParsingException {
        // process any keys in base response
        if (key.equals(ProtocolConstants.KEY_RESULT_CODE)) {
            response.setResultCode(parseResultCodeValue(value));
            return true;
        } else if (key.equals(ProtocolConstants.KEY_MESSAGE)) {
            response.setMessage(value);
            return true;
        } else if (key.equals(ProtocolConstants.KEY_VERSION)) {
            response.setVersion(parseVersionValue(value));
            return true;
        } else {
            return false;
        }
    }

    static public MonitorResponse parseMonitorResponse(String body) throws ProtocolParsingException {
        // create an empty response
        MonitorResponse response = new MonitorResponse();

        // split body into lines
        String[] lines = splitBody(body);

        // process each line in body
        for (int i = 0; i < lines.length; i++) {
            // parse each line into key/value pairs
            String[] kv = splitKeyValue(lines[i]);
            String key = kv[0];
            String value = kv[1];

            // parse expected keys
            if (handleBaseResponseKeys(response, key, value)) {
                // already processed, keep processing
            } else if (key.equals(ProtocolConstants.KEY_AREA_ID)) {
                response.setAreaId(parseAreaIdValue(value));
            } else if (key.equals(ProtocolConstants.KEY_QUEUE)) {
                processQueueValue(value, response);
            } else {
                logger.warn("Unsupported key [" + key + "] in monitor response line [" + lines[i] + "]");
            }
        }

        // was there at least a resultCode?
        if (response.getResultCode() < 0) {
            throw new ProtocolParsingException("Monitor response did not contain a valid resultCode");
        }

        return response;
    }

    static public TransferResponse parseTransferResponse(String body) throws ProtocolParsingException {
        // create an empty response
        TransferResponse response = new TransferResponse();

        // split body into lines
        String[] lines = splitBody(body);

        // process each line in body
        for (int i = 0; i < lines.length; i++) {
            // parse each line into key/value pairs
            String[] kv = splitKeyValue(lines[i]);
            String key = kv[0];
            String value = kv[1];

            // parse expected keys
            if (handleBaseResponseKeys(response, key, value)) {
                // already processed, keep processing
            } else {
                logger.warn("Unsupported key [" + key + "] in transfer response line [" + lines[i] + "]");
            }
        }

        // was there at least a resultCode?
        if (response.getResultCode() < 0) {
            throw new ProtocolParsingException("Transfer response did not contain a valid resultCode");
        }

        return response;
    }

}
