package com.cloudhopper.mq.broker.server;

/*
 * #%L
 * ch-mq-remote
 * %%
 * Copyright (C) 2009 - 2012 Cloudhopper by Twitter
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.cloudhopper.mq.broker.DistributedQueueManager;
import com.cloudhopper.mq.queue.QueueManager;
import javax.management.ObjectName;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * HTTP Server as a "Broker".
 * 
 * @version $Revision$
 */
public class BrokerServer implements BrokerServerMBean {
    private static final Logger logger = LoggerFactory.getLogger(BrokerServer.class);

    private final BrokerServerConfiguration configuration;
    private final BrokerServlet brokerServlet;
    private final StatusServlet statusServlet;
    private final ObserveServlet observeServlet;
    private final QueueManager queueManager;
    private final DistributedQueueManager dqm;
    private JettyHttpServer jetty;

    public BrokerServer(BrokerServerConfiguration configuration, QueueManager queueManager, DistributedQueueManager dqm) {
        this.configuration = configuration;
        this.queueManager = queueManager;
        this.dqm = dqm;
        this.brokerServlet = new BrokerServlet(queueManager, dqm);
        this.statusServlet = new StatusServlet(queueManager, dqm);
        this.observeServlet = new ObserveServlet(queueManager, dqm);
        registerMBean();
    }

    protected void registerMBean() {
        if (queueManager.getConfiguration().isJmxEnabled()) {
            // register the this queue manager as an mbean
            try {
                ObjectName name = new ObjectName(queueManager.getConfiguration().getJmxDomain() + ":name=BrokerServer");
                queueManager.getConfiguration().getMBeanServer().registerMBean(this, name);
            } catch (Exception e) {
                // log the error, but don't throw an exception for this datasource
                logger.warn("Error while attempting to register BrokerServer as an MBean: {}", e.toString());
            }
        }
    }

    /**
     * Overrides the default "put" handler for the BrokerServlet.
     * @param putHandler
     */
    public void setPutHandler(BrokerServerPutHandler putHandler) {
        this.brokerServlet.setPutHandler(putHandler);
    }

    public BrokerServerConfiguration getConfiguration() {
        return this.configuration;
    }

    public int getPort() {
        return this.configuration.getPort();
    }

    public int getMinThreads() {
        return this.configuration.getMinThreads();
    }

    public int getMaxThreads() {
        return this.configuration.getMaxThreads();
    }

    public boolean getUseDirectBuffers() {
        return this.configuration.getUseDirectBuffers();
    }
    
    public boolean isStarted() {
	return (jetty != null && jetty.isRunning());
    }

    public boolean isStopped() {
	return (jetty == null || !jetty.isRunning());
    }

    synchronized public void start() throws Exception {
        if (isStarted()) {
            logger.warn("Ignoring start request, already started");
            return;
        }

	jetty = new JettyHttpServer();
	jetty.setPort(getPort());
	jetty.setMinThreads(getMinThreads());
	jetty.setMaxThreads(getMaxThreads());
	jetty.setUseDirectBuffers(getUseDirectBuffers());
	jetty.start();

	jetty.addServlet(new NotFound404Servlet(), "Not_Found_Servlet", "/*");
	jetty.addServlet(statusServlet, "Status_Servlet", "/status");
	jetty.addServlet(brokerServlet, "Broker_Servlet", "/broker");
        jetty.addServlet(observeServlet, "Observe_Servlet", "/observe");
    }

    synchronized public void stop() throws Exception {
        if (!isStarted()) {
            logger.warn("Ignoring stop request, already stopped");
            return;
        }

        logger.info("Attempting to stop broker HTTP server...");

	jetty.stop();
    }

    public long getThreadPoolWaitingTaskCount() {
	return -1; //Unsupported
    }

    public long getThreadPoolSubmittedTaskCount() {
	return -1; //Unsupported
    }

    public long getThreadPoolCompletedTaskCount() {
	return -1; //Unsupported
    }

    public int getThreadPoolBusyThreadCount() {
	return -1; //Unsupported
    }

    public int getThreadPoolIdleThreadCount() {
	return jetty.getThreadPool().getIdleThreads();
    }

    public int getThreadPoolCurrentThreadCount() {
	return jetty.getThreadPool().getThreads();
    }

    public int getThreadPoolMinThreadCount() {
	return jetty.getThreadPool().getMinThreads();
    }

    public int getThreadPoolMaxThreadCount() {
	return jetty.getThreadPool().getMaxThreads();
    }

    public int getThreadPoolPeakThreadCount() {
	return -1; //Unsupported
    }

}
