;; Copyright (c) Cognitect, Inc.
;; All rights reserved.

(ns cognitect.aws.lambda.specs
  (:require [clojure.spec.alpha :as s] [clojure.spec.gen.alpha :as gen]))

(s/def :cognitect.aws/client map?)

(s/def :core.async/channel any?)

(s/def
  :cognitect.aws.lambda/BatchSize
  (s/with-gen
    (s/and int? #(<= 1 % 10000))
    #(gen/choose (or 1 Long/MIN_VALUE) (or 10000 Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/GetLayerVersionPolicyRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.GetLayerVersionPolicyRequest/LayerName
     :cognitect.aws.lambda.GetLayerVersionPolicyRequest/VersionNumber]))

(s/def :cognitect.aws.lambda/SubnetIds (s/coll-of :cognitect.aws.lambda/SubnetId :max-count 16))

(s/def
  :cognitect.aws.lambda/ListLayerVersionsRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.ListLayerVersionsRequest/LayerName]
    :opt-un
    [:cognitect.aws.lambda.ListLayerVersionsRequest/Marker
     :cognitect.aws.lambda.ListLayerVersionsRequest/CompatibleRuntime
     :cognitect.aws.lambda.ListLayerVersionsRequest/MaxItems]))

(s/def :cognitect.aws.lambda/Principal (s/with-gen #(re-matches (re-pattern ".*") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/TagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.TagResourceRequest/Resource
     :cognitect.aws.lambda.TagResourceRequest/Tags]))

(s/def
  :cognitect.aws.lambda/RemoveLayerVersionPermissionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.RemoveLayerVersionPermissionRequest/LayerName
     :cognitect.aws.lambda.RemoveLayerVersionPermissionRequest/VersionNumber
     :cognitect.aws.lambda.RemoveLayerVersionPermissionRequest/StatementId]
    :opt-un
    [:cognitect.aws.lambda.RemoveLayerVersionPermissionRequest/RevisionId]))

(s/def
  :cognitect.aws.lambda/CompatibleRuntimes
  (s/coll-of :cognitect.aws.lambda/Runtime :max-count 5))

(s/def
  :cognitect.aws.lambda/PublishLayerVersionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.PublishLayerVersionRequest/LayerName
     :cognitect.aws.lambda.PublishLayerVersionRequest/Content]
    :opt-un
    [:cognitect.aws.lambda.PublishLayerVersionRequest/LicenseInfo
     :cognitect.aws.lambda.PublishLayerVersionRequest/Description
     :cognitect.aws.lambda.PublishLayerVersionRequest/CompatibleRuntimes]))

(s/def
  :cognitect.aws.lambda/LayerVersionContentInput
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.LayerVersionContentInput/S3ObjectVersion
     :cognitect.aws.lambda.LayerVersionContentInput/S3Key
     :cognitect.aws.lambda.LayerVersionContentInput/S3Bucket
     :cognitect.aws.lambda.LayerVersionContentInput/ZipFile]))

(s/def
  :cognitect.aws.lambda/ListTagsRequest
  (s/keys :req-un [:cognitect.aws.lambda.ListTagsRequest/Resource]))

(s/def :cognitect.aws.lambda/LayerList (s/coll-of :cognitect.aws.lambda/LayerVersionArn))

(s/def :cognitect.aws.lambda/LogType string?)

(s/def
  :cognitect.aws.lambda/GetFunctionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.GetFunctionRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.GetFunctionRequest/Qualifier]))

(s/def
  :cognitect.aws.lambda/Environment
  (s/keys :opt-un [:cognitect.aws.lambda.Environment/Variables]))

(s/def
  :cognitect.aws.lambda/UpdateFunctionCodeRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.UpdateFunctionCodeRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.UpdateFunctionCodeRequest/DryRun
     :cognitect.aws.lambda.UpdateFunctionCodeRequest/S3ObjectVersion
     :cognitect.aws.lambda.UpdateFunctionCodeRequest/S3Key
     :cognitect.aws.lambda.UpdateFunctionCodeRequest/S3Bucket
     :cognitect.aws.lambda.UpdateFunctionCodeRequest/RevisionId
     :cognitect.aws.lambda.UpdateFunctionCodeRequest/ZipFile
     :cognitect.aws.lambda.UpdateFunctionCodeRequest/Publish]))

(s/def
  :cognitect.aws.lambda/GetEventSourceMappingRequest
  (s/keys :req-un [:cognitect.aws.lambda.GetEventSourceMappingRequest/UUID]))

(s/def
  :cognitect.aws.lambda/AddLayerVersionPermissionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.AddLayerVersionPermissionRequest/LayerName
     :cognitect.aws.lambda.AddLayerVersionPermissionRequest/VersionNumber
     :cognitect.aws.lambda.AddLayerVersionPermissionRequest/StatementId
     :cognitect.aws.lambda.AddLayerVersionPermissionRequest/Action
     :cognitect.aws.lambda.AddLayerVersionPermissionRequest/Principal]
    :opt-un
    [:cognitect.aws.lambda.AddLayerVersionPermissionRequest/OrganizationId
     :cognitect.aws.lambda.AddLayerVersionPermissionRequest/RevisionId]))

(s/def
  :cognitect.aws.lambda/Timeout
  (s/with-gen (s/and int? #(<= 1 %)) #(gen/choose (or 1 Long/MIN_VALUE) (or nil Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/InvokeAsyncResponse
  (s/keys :opt-un [:cognitect.aws.lambda.InvokeAsyncResponse/Status]))

(s/def :cognitect.aws.lambda/EnvironmentVariableValue string?)

(s/def
  :cognitect.aws.lambda/InvocationResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.InvocationResponse/FunctionError
     :cognitect.aws.lambda.InvocationResponse/StatusCode
     :cognitect.aws.lambda.InvocationResponse/ExecutedVersion
     :cognitect.aws.lambda.InvocationResponse/Payload
     :cognitect.aws.lambda.InvocationResponse/LogResult]))

(s/def
  :cognitect.aws.lambda/StatementId
  (s/with-gen #(re-matches (re-pattern "([a-zA-Z0-9-_]+)") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/DeleteFunctionConcurrencyRequest
  (s/keys :req-un [:cognitect.aws.lambda.DeleteFunctionConcurrencyRequest/FunctionName]))

(s/def
  :cognitect.aws.lambda/ListAliasesRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.ListAliasesRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.ListAliasesRequest/Marker
     :cognitect.aws.lambda.ListAliasesRequest/FunctionVersion
     :cognitect.aws.lambda.ListAliasesRequest/MaxItems]))

(s/def
  :cognitect.aws.lambda/KMSKeyArn
  (s/with-gen
    #(re-matches (re-pattern "(arn:(aws[a-zA-Z-]*)?:[a-z0-9-.]+:.*)|()") %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/GetFunctionConfigurationRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.GetFunctionConfigurationRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.GetFunctionConfigurationRequest/Qualifier]))

(s/def
  :cognitect.aws.lambda/ListVersionsByFunctionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.ListVersionsByFunctionRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.ListVersionsByFunctionRequest/Marker
     :cognitect.aws.lambda.ListVersionsByFunctionRequest/MaxItems]))

(s/def
  :cognitect.aws.lambda/LayersListItem
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.LayersListItem/LatestMatchingVersion
     :cognitect.aws.lambda.LayersListItem/LayerName
     :cognitect.aws.lambda.LayersListItem/LayerArn]))

(s/def
  :cognitect.aws.lambda/VpcConfig
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.VpcConfig/SubnetIds :cognitect.aws.lambda.VpcConfig/SecurityGroupIds]))

(s/def
  :cognitect.aws.lambda/NamespacedFunctionName
  (s/with-gen
    #(re-matches
      (re-pattern
        "(arn:(aws[a-zA-Z-]*)?:lambda:)?([a-z]{2}(-gov)?-[a-z]+-\\d{1}:)?(\\d{12}:)?(function:)?([a-zA-Z0-9-_\\.]+)(:(\\$LATEST|[a-zA-Z0-9-_]+))?")
      %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/FunctionConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.FunctionConfiguration/TracingConfig
     :cognitect.aws.lambda.FunctionConfiguration/FunctionArn
     :cognitect.aws.lambda.FunctionConfiguration/FunctionName
     :cognitect.aws.lambda.FunctionConfiguration/Timeout
     :cognitect.aws.lambda.FunctionConfiguration/Layers
     :cognitect.aws.lambda.FunctionConfiguration/DeadLetterConfig
     :cognitect.aws.lambda.FunctionConfiguration/Handler
     :cognitect.aws.lambda.FunctionConfiguration/Runtime
     :cognitect.aws.lambda.FunctionConfiguration/MemorySize
     :cognitect.aws.lambda.FunctionConfiguration/Role
     :cognitect.aws.lambda.FunctionConfiguration/KMSKeyArn
     :cognitect.aws.lambda.FunctionConfiguration/Description
     :cognitect.aws.lambda.FunctionConfiguration/LastModified
     :cognitect.aws.lambda.FunctionConfiguration/Environment
     :cognitect.aws.lambda.FunctionConfiguration/CodeSize
     :cognitect.aws.lambda.FunctionConfiguration/VpcConfig
     :cognitect.aws.lambda.FunctionConfiguration/RevisionId
     :cognitect.aws.lambda.FunctionConfiguration/Version
     :cognitect.aws.lambda.FunctionConfiguration/CodeSha256
     :cognitect.aws.lambda.FunctionConfiguration/MasterArn]))

(s/def
  :cognitect.aws.lambda/GetAccountSettingsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.GetAccountSettingsResponse/AccountUsage
     :cognitect.aws.lambda.GetAccountSettingsResponse/AccountLimit]))

(s/def
  :cognitect.aws.lambda/AdditionalVersionWeights
  (s/map-of :cognitect.aws.lambda/AdditionalVersion :cognitect.aws.lambda/Weight))

(s/def
  :cognitect.aws.lambda/LayerVersionArn
  (s/with-gen
    #(re-matches
      (re-pattern "arn:[a-zA-Z0-9-]+:lambda:[a-zA-Z0-9-]+:\\d{12}:layer:[a-zA-Z0-9-_]+:[0-9]+")
      %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/S3ObjectVersion
  (s/with-gen
    (s/and string? #(<= 1 (count %) 1024))
    (fn []
      (gen/fmap
        #(apply str %)
        (gen/bind (gen/choose (or 1 0) (or 1024 100)) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.lambda/GetLayerVersionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.GetLayerVersionRequest/LayerName
     :cognitect.aws.lambda.GetLayerVersionRequest/VersionNumber]))

(s/def
  :cognitect.aws.lambda/UpdateFunctionConfigurationRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.UpdateFunctionConfigurationRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.UpdateFunctionConfigurationRequest/TracingConfig
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Timeout
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Layers
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/DeadLetterConfig
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Handler
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Runtime
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/MemorySize
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Role
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/KMSKeyArn
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Description
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Environment
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/VpcConfig
     :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/RevisionId]))

(s/def
  :cognitect.aws.lambda/NameSpacedFunctionArn
  (s/with-gen
    #(re-matches
      (re-pattern
        "arn:(aws[a-zA-Z-]*)?:lambda:[a-z]{2}(-gov)?-[a-z]+-\\d{1}:\\d{12}:function:[a-zA-Z0-9-_\\.]+(:(\\$LATEST|[a-zA-Z0-9-_]+))?")
      %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/ListAliasesResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListAliasesResponse/Aliases
     :cognitect.aws.lambda.ListAliasesResponse/NextMarker]))

(s/def
  :cognitect.aws.lambda/PublishLayerVersionResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.PublishLayerVersionResponse/LicenseInfo
     :cognitect.aws.lambda.PublishLayerVersionResponse/Description
     :cognitect.aws.lambda.PublishLayerVersionResponse/CreatedDate
     :cognitect.aws.lambda.PublishLayerVersionResponse/Content
     :cognitect.aws.lambda.PublishLayerVersionResponse/CompatibleRuntimes
     :cognitect.aws.lambda.PublishLayerVersionResponse/LayerVersionArn
     :cognitect.aws.lambda.PublishLayerVersionResponse/Version
     :cognitect.aws.lambda.PublishLayerVersionResponse/LayerArn]))

(s/def
  :cognitect.aws.lambda/PublishVersionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.PublishVersionRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.PublishVersionRequest/Description
     :cognitect.aws.lambda.PublishVersionRequest/RevisionId
     :cognitect.aws.lambda.PublishVersionRequest/CodeSha256]))

(s/def
  :cognitect.aws.lambda/MaxLayerListItems
  (s/with-gen (s/and int? #(<= 1 % 50)) #(gen/choose (or 1 Long/MIN_VALUE) (or 50 Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/UpdateAliasRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.UpdateAliasRequest/FunctionName
     :cognitect.aws.lambda.UpdateAliasRequest/Name]
    :opt-un
    [:cognitect.aws.lambda.UpdateAliasRequest/RoutingConfig
     :cognitect.aws.lambda.UpdateAliasRequest/FunctionVersion
     :cognitect.aws.lambda.UpdateAliasRequest/Description
     :cognitect.aws.lambda.UpdateAliasRequest/RevisionId]))

(s/def
  :cognitect.aws.lambda/FunctionCode
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.FunctionCode/S3ObjectVersion
     :cognitect.aws.lambda.FunctionCode/S3Key
     :cognitect.aws.lambda.FunctionCode/S3Bucket
     :cognitect.aws.lambda.FunctionCode/ZipFile]))

(s/def
  :cognitect.aws.lambda/DeadLetterConfig
  (s/keys :opt-un [:cognitect.aws.lambda.DeadLetterConfig/TargetArn]))

(s/def
  :cognitect.aws.lambda/Tags
  (s/map-of :cognitect.aws.lambda/TagKey :cognitect.aws.lambda/TagValue))

(s/def
  :cognitect.aws.lambda/LayerName
  (s/with-gen
    #(re-matches
      (re-pattern
        "(arn:[a-zA-Z0-9-]+:lambda:[a-zA-Z0-9-]+:\\d{12}:layer:[a-zA-Z0-9-_]+)|[a-zA-Z0-9-_]+")
      %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/MaximumBatchingWindowInSeconds
  (s/with-gen
    (s/and int? #(<= 0 % 300))
    #(gen/choose (or 0 Long/MIN_VALUE) (or 300 Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/AliasConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.AliasConfiguration/RoutingConfig
     :cognitect.aws.lambda.AliasConfiguration/FunctionVersion
     :cognitect.aws.lambda.AliasConfiguration/Description
     :cognitect.aws.lambda.AliasConfiguration/RevisionId
     :cognitect.aws.lambda.AliasConfiguration/Name
     :cognitect.aws.lambda.AliasConfiguration/AliasArn]))

(s/def
  :cognitect.aws.lambda/EventSourceToken
  (s/with-gen #(re-matches (re-pattern "[a-zA-Z0-9._\\-]+") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/FunctionName
  (s/with-gen
    #(re-matches
      (re-pattern
        "(arn:(aws[a-zA-Z-]*)?:lambda:)?([a-z]{2}(-gov)?-[a-z]+-\\d{1}:)?(\\d{12}:)?(function:)?([a-zA-Z0-9-_]+)(:(\\$LATEST|[a-zA-Z0-9-_]+))?")
      %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/S3Key
  (s/with-gen
    (s/and string? #(<= 1 (count %) 1024))
    (fn []
      (gen/fmap
        #(apply str %)
        (gen/bind (gen/choose (or 1 0) (or 1024 100)) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.lambda/ListLayersResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListLayersResponse/Layers
     :cognitect.aws.lambda.ListLayersResponse/NextMarker]))

(s/def
  :cognitect.aws.lambda/GetLayerVersionByArnRequest
  (s/keys :req-un [:cognitect.aws.lambda.GetLayerVersionByArnRequest/Arn]))

(s/def
  :cognitect.aws.lambda/UnreservedConcurrentExecutions
  (s/with-gen (s/and int? #(<= 0 %)) #(gen/choose (or 0 Long/MIN_VALUE) (or nil Long/MAX_VALUE))))

(s/def :cognitect.aws.lambda/Boolean boolean?)

(s/def
  :cognitect.aws.lambda/ListLayersRequest
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListLayersRequest/Marker
     :cognitect.aws.lambda.ListLayersRequest/CompatibleRuntime
     :cognitect.aws.lambda.ListLayersRequest/MaxItems]))

(s/def
  :cognitect.aws.lambda/DeleteEventSourceMappingRequest
  (s/keys :req-un [:cognitect.aws.lambda.DeleteEventSourceMappingRequest/UUID]))

(s/def
  :cognitect.aws.lambda/S3Bucket
  (s/with-gen #(re-matches (re-pattern "^[0-9A-Za-z\\.\\-_]*(?<!\\.)$") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/EnvironmentError
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.EnvironmentError/ErrorCode
     :cognitect.aws.lambda.EnvironmentError/Message]))

(s/def
  :cognitect.aws.lambda/Integer
  (s/with-gen int? #(gen/choose (or nil Long/MIN_VALUE) (or nil Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/ListTagsResponse
  (s/keys :opt-un [:cognitect.aws.lambda.ListTagsResponse/Tags]))

(s/def
  :cognitect.aws.lambda/InvocationRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.InvocationRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.InvocationRequest/Payload
     :cognitect.aws.lambda.InvocationRequest/LogType
     :cognitect.aws.lambda.InvocationRequest/ClientContext
     :cognitect.aws.lambda.InvocationRequest/InvocationType
     :cognitect.aws.lambda.InvocationRequest/Qualifier]))

(s/def :cognitect.aws.lambda/TagValue string?)

(s/def
  :cognitect.aws.lambda/AccountUsage
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.AccountUsage/TotalCodeSize
     :cognitect.aws.lambda.AccountUsage/FunctionCount]))

(s/def
  :cognitect.aws.lambda/LayerVersionNumber
  (s/with-gen int? #(gen/choose (or nil Long/MIN_VALUE) (or nil Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/HttpStatus
  (s/with-gen int? #(gen/choose (or nil Long/MIN_VALUE) (or nil Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/NamespacedStatementId
  (s/with-gen #(re-matches (re-pattern "([a-zA-Z0-9-_.]+)") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/ListFunctionsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListFunctionsRequest/Marker
     :cognitect.aws.lambda.ListFunctionsRequest/FunctionVersion
     :cognitect.aws.lambda.ListFunctionsRequest/MasterRegion
     :cognitect.aws.lambda.ListFunctionsRequest/MaxItems]))

(s/def
  :cognitect.aws.lambda/ListLayerVersionsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListLayerVersionsResponse/LayerVersions
     :cognitect.aws.lambda.ListLayerVersionsResponse/NextMarker]))

(s/def
  :cognitect.aws.lambda/TracingConfig
  (s/keys :opt-un [:cognitect.aws.lambda.TracingConfig/Mode]))

(s/def
  :cognitect.aws.lambda/RoleArn
  (s/with-gen
    #(re-matches (re-pattern "arn:(aws[a-zA-Z-]*)?:iam::\\d{12}:role/?[a-zA-Z_0-9+=,.@\\-_/]+") %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/LayerVersionsListItem
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.LayerVersionsListItem/LicenseInfo
     :cognitect.aws.lambda.LayerVersionsListItem/Description
     :cognitect.aws.lambda.LayerVersionsListItem/CreatedDate
     :cognitect.aws.lambda.LayerVersionsListItem/CompatibleRuntimes
     :cognitect.aws.lambda.LayerVersionsListItem/LayerVersionArn
     :cognitect.aws.lambda.LayerVersionsListItem/Version]))

(s/def
  :cognitect.aws.lambda/EventSourceMappingsList
  (s/coll-of :cognitect.aws.lambda/EventSourceMappingConfiguration))

(s/def
  :cognitect.aws.lambda/FunctionArn
  (s/with-gen
    #(re-matches
      (re-pattern
        "arn:(aws[a-zA-Z-]*)?:lambda:[a-z]{2}(-gov)?-[a-z]+-\\d{1}:\\d{12}:function:[a-zA-Z0-9-_]+(:(\\$LATEST|[a-zA-Z0-9-_]+))?")
      %)
    #(gen/string)))

(s/def :cognitect.aws.lambda/SecurityGroupId string?)

(s/def :cognitect.aws.lambda/SubnetId string?)

(s/def
  :cognitect.aws.lambda/LicenseInfo
  (s/with-gen
    (s/and string? #(>= 512 (count %)))
    (fn []
      (gen/fmap
        #(apply str %)
        (gen/bind (gen/choose (or nil 0) (or 512 100)) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.lambda/EnvironmentVariables
  (s/map-of
    :cognitect.aws.lambda/EnvironmentVariableName
    :cognitect.aws.lambda/EnvironmentVariableValue))

(s/def
  :cognitect.aws.lambda/GetLayerVersionPolicyResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.GetLayerVersionPolicyResponse/RevisionId
     :cognitect.aws.lambda.GetLayerVersionPolicyResponse/Policy]))

(s/def
  :cognitect.aws.lambda/AdditionalVersion
  (s/with-gen #(re-matches (re-pattern "[0-9]+") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/SourceOwner
  (s/with-gen #(re-matches (re-pattern "\\d{12}") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/CreateAliasRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.CreateAliasRequest/FunctionName
     :cognitect.aws.lambda.CreateAliasRequest/Name
     :cognitect.aws.lambda.CreateAliasRequest/FunctionVersion]
    :opt-un
    [:cognitect.aws.lambda.CreateAliasRequest/RoutingConfig
     :cognitect.aws.lambda.CreateAliasRequest/Description]))

(s/def
  :cognitect.aws.lambda/ListVersionsByFunctionResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListVersionsByFunctionResponse/Versions
     :cognitect.aws.lambda.ListVersionsByFunctionResponse/NextMarker]))

(s/def
  :cognitect.aws.lambda/ListEventSourceMappingsRequest
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListEventSourceMappingsRequest/FunctionName
     :cognitect.aws.lambda.ListEventSourceMappingsRequest/EventSourceArn
     :cognitect.aws.lambda.ListEventSourceMappingsRequest/Marker
     :cognitect.aws.lambda.ListEventSourceMappingsRequest/MaxItems]))

(s/def
  :cognitect.aws.lambda/LayerPermissionAllowedAction
  (s/with-gen #(re-matches (re-pattern "lambda:GetLayerVersion") %) #(gen/string)))

(s/def :cognitect.aws.lambda/EventSourcePosition string?)

(s/def
  :cognitect.aws.lambda/DeleteFunctionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.DeleteFunctionRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.DeleteFunctionRequest/Qualifier]))

(s/def
  :cognitect.aws.lambda/CreateFunctionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.CreateFunctionRequest/FunctionName
     :cognitect.aws.lambda.CreateFunctionRequest/Runtime
     :cognitect.aws.lambda.CreateFunctionRequest/Role
     :cognitect.aws.lambda.CreateFunctionRequest/Handler
     :cognitect.aws.lambda.CreateFunctionRequest/Code]
    :opt-un
    [:cognitect.aws.lambda.CreateFunctionRequest/TracingConfig
     :cognitect.aws.lambda.CreateFunctionRequest/Timeout
     :cognitect.aws.lambda.CreateFunctionRequest/Layers
     :cognitect.aws.lambda.CreateFunctionRequest/DeadLetterConfig
     :cognitect.aws.lambda.CreateFunctionRequest/Tags
     :cognitect.aws.lambda.CreateFunctionRequest/MemorySize
     :cognitect.aws.lambda.CreateFunctionRequest/KMSKeyArn
     :cognitect.aws.lambda.CreateFunctionRequest/Description
     :cognitect.aws.lambda.CreateFunctionRequest/Environment
     :cognitect.aws.lambda.CreateFunctionRequest/VpcConfig
     :cognitect.aws.lambda.CreateFunctionRequest/Publish]))

(s/def :cognitect.aws.lambda/Date inst?)

(s/def
  :cognitect.aws.lambda/VpcConfigResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.VpcConfigResponse/SubnetIds
     :cognitect.aws.lambda.VpcConfigResponse/SecurityGroupIds
     :cognitect.aws.lambda.VpcConfigResponse/VpcId]))

(s/def
  :cognitect.aws.lambda/AddPermissionResponse
  (s/keys :opt-un [:cognitect.aws.lambda.AddPermissionResponse/Statement]))

(s/def
  :cognitect.aws.lambda/ReservedConcurrentExecutions
  (s/with-gen (s/and int? #(<= 0 %)) #(gen/choose (or 0 Long/MIN_VALUE) (or nil Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/Action
  (s/with-gen #(re-matches (re-pattern "(lambda:[*]|lambda:[a-zA-Z]+|[*])") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/Long
  (s/with-gen int? #(gen/choose (or nil Long/MIN_VALUE) (or nil Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/Arn
  (s/with-gen
    #(re-matches
      (re-pattern
        "arn:(aws[a-zA-Z0-9-]*):([a-zA-Z0-9\\-])+:([a-z]{2}(-gov)?-[a-z]+-\\d{1})?:(\\d{12})?:(.*)")
      %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/MasterRegion
  (s/with-gen #(re-matches (re-pattern "ALL|[a-z]{2}(-gov)?-[a-z]+-\\d{1}") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/CreateEventSourceMappingRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.CreateEventSourceMappingRequest/EventSourceArn
     :cognitect.aws.lambda.CreateEventSourceMappingRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.CreateEventSourceMappingRequest/BatchSize
     :cognitect.aws.lambda.CreateEventSourceMappingRequest/MaximumBatchingWindowInSeconds
     :cognitect.aws.lambda.CreateEventSourceMappingRequest/Enabled
     :cognitect.aws.lambda.CreateEventSourceMappingRequest/StartingPositionTimestamp
     :cognitect.aws.lambda.CreateEventSourceMappingRequest/StartingPosition]))

(s/def
  :cognitect.aws.lambda/AddPermissionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.AddPermissionRequest/FunctionName
     :cognitect.aws.lambda.AddPermissionRequest/StatementId
     :cognitect.aws.lambda.AddPermissionRequest/Action
     :cognitect.aws.lambda.AddPermissionRequest/Principal]
    :opt-un
    [:cognitect.aws.lambda.AddPermissionRequest/SourceAccount
     :cognitect.aws.lambda.AddPermissionRequest/SourceArn
     :cognitect.aws.lambda.AddPermissionRequest/RevisionId
     :cognitect.aws.lambda.AddPermissionRequest/Qualifier
     :cognitect.aws.lambda.AddPermissionRequest/EventSourceToken]))

(s/def
  :cognitect.aws.lambda/PutFunctionConcurrencyRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.PutFunctionConcurrencyRequest/FunctionName
     :cognitect.aws.lambda.PutFunctionConcurrencyRequest/ReservedConcurrentExecutions]))

(s/def :cognitect.aws.lambda/TracingMode string?)

(s/def
  :cognitect.aws.lambda/FunctionCodeLocation
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.FunctionCodeLocation/RepositoryType
     :cognitect.aws.lambda.FunctionCodeLocation/Location]))

(s/def
  :cognitect.aws.lambda/GetPolicyResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.GetPolicyResponse/RevisionId
     :cognitect.aws.lambda.GetPolicyResponse/Policy]))

(s/def
  :cognitect.aws.lambda/UpdateEventSourceMappingRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.UpdateEventSourceMappingRequest/UUID]
    :opt-un
    [:cognitect.aws.lambda.UpdateEventSourceMappingRequest/BatchSize
     :cognitect.aws.lambda.UpdateEventSourceMappingRequest/FunctionName
     :cognitect.aws.lambda.UpdateEventSourceMappingRequest/MaximumBatchingWindowInSeconds
     :cognitect.aws.lambda.UpdateEventSourceMappingRequest/Enabled]))

(s/def
  :cognitect.aws.lambda/DeleteLayerVersionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.DeleteLayerVersionRequest/LayerName
     :cognitect.aws.lambda.DeleteLayerVersionRequest/VersionNumber]))

(s/def
  :cognitect.aws.lambda/LayerArn
  (s/with-gen
    #(re-matches
      (re-pattern "arn:[a-zA-Z0-9-]+:lambda:[a-zA-Z0-9-]+:\\d{12}:layer:[a-zA-Z0-9-_]+")
      %)
    #(gen/string)))

(s/def :cognitect.aws.lambda/LayersReferenceList (s/coll-of :cognitect.aws.lambda/Layer))

(s/def
  :cognitect.aws.lambda/UntagResourceRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.UntagResourceRequest/Resource
     :cognitect.aws.lambda.UntagResourceRequest/TagKeys]))

(s/def :cognitect.aws.lambda/TagKey string?)

(s/def :cognitect.aws.lambda/InvocationType string?)

(s/def
  :cognitect.aws.lambda/DeleteAliasRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.DeleteAliasRequest/FunctionName
     :cognitect.aws.lambda.DeleteAliasRequest/Name]))

(s/def
  :cognitect.aws.lambda/Blob
  (s/or :byte-array bytes? :input-stream #(instance? java.io.InputStream %)))

(s/def
  :cognitect.aws.lambda/Concurrency
  (s/keys :opt-un [:cognitect.aws.lambda.Concurrency/ReservedConcurrentExecutions]))

(s/def
  :cognitect.aws.lambda/InvokeAsyncRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.InvokeAsyncRequest/FunctionName
     :cognitect.aws.lambda.InvokeAsyncRequest/InvokeArgs]))

(s/def
  :cognitect.aws.lambda/Qualifier
  (s/with-gen #(re-matches (re-pattern "(|[a-zA-Z0-9$_-]+)") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/Description
  (s/with-gen
    (s/and string? #(<= 0 (count %) 256))
    (fn []
      (gen/fmap
        #(apply str %)
        (gen/bind (gen/choose (or 0 0) (or 256 100)) #(gen/vector (gen/char-alpha) %))))))

(s/def
  :cognitect.aws.lambda/GetAliasRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.GetAliasRequest/FunctionName
     :cognitect.aws.lambda.GetAliasRequest/Name]))

(s/def :cognitect.aws.lambda/Enabled boolean?)

(s/def
  :cognitect.aws.lambda/EnvironmentVariableName
  (s/with-gen #(re-matches (re-pattern "[a-zA-Z]([a-zA-Z0-9_])+") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/LayerVersionContentOutput
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.LayerVersionContentOutput/Location
     :cognitect.aws.lambda.LayerVersionContentOutput/CodeSize
     :cognitect.aws.lambda.LayerVersionContentOutput/CodeSha256]))

(s/def :cognitect.aws.lambda/Timestamp string?)

(s/def
  :cognitect.aws.lambda/GetLayerVersionResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.GetLayerVersionResponse/LicenseInfo
     :cognitect.aws.lambda.GetLayerVersionResponse/Description
     :cognitect.aws.lambda.GetLayerVersionResponse/CreatedDate
     :cognitect.aws.lambda.GetLayerVersionResponse/Content
     :cognitect.aws.lambda.GetLayerVersionResponse/CompatibleRuntimes
     :cognitect.aws.lambda.GetLayerVersionResponse/LayerVersionArn
     :cognitect.aws.lambda.GetLayerVersionResponse/Version
     :cognitect.aws.lambda.GetLayerVersionResponse/LayerArn]))

(s/def :cognitect.aws.lambda/Runtime string?)

(s/def
  :cognitect.aws.lambda/MaxListItems
  (s/with-gen
    (s/and int? #(<= 1 % 10000))
    #(gen/choose (or 1 Long/MIN_VALUE) (or 10000 Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/EnvironmentResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.EnvironmentResponse/Variables
     :cognitect.aws.lambda.EnvironmentResponse/Error]))

(s/def
  :cognitect.aws.lambda/ResourceArn
  (s/with-gen
    #(re-matches (re-pattern "(arn:(aws[a-zA-Z-]*)?:[a-z0-9-.]+:.*)|()") %)
    #(gen/string)))

(s/def :cognitect.aws.lambda/FunctionVersion string?)

(s/def :cognitect.aws.lambda/AliasList (s/coll-of :cognitect.aws.lambda/AliasConfiguration))

(s/def
  :cognitect.aws.lambda/OrganizationId
  (s/with-gen #(re-matches (re-pattern "o-[a-z0-9]{10,32}") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/Alias
  (s/with-gen #(re-matches (re-pattern "(?!^[0-9]+$)([a-zA-Z0-9-_]+)") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/LayerPermissionAllowedPrincipal
  (s/with-gen
    #(re-matches (re-pattern "\\d{12}|\\*|arn:(aws[a-zA-Z-]*):iam::\\d{12}:root") %)
    #(gen/string)))

(s/def
  :cognitect.aws.lambda/AliasRoutingConfiguration
  (s/keys :opt-un [:cognitect.aws.lambda.AliasRoutingConfiguration/AdditionalVersionWeights]))

(s/def
  :cognitect.aws.lambda/MemorySize
  (s/with-gen
    (s/and int? #(<= 128 % 3008))
    #(gen/choose (or 128 Long/MIN_VALUE) (or 3008 Long/MAX_VALUE))))

(s/def
  :cognitect.aws.lambda/SecurityGroupIds
  (s/coll-of :cognitect.aws.lambda/SecurityGroupId :max-count 5))

(s/def
  :cognitect.aws.lambda/AccountLimit
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.AccountLimit/UnreservedConcurrentExecutions
     :cognitect.aws.lambda.AccountLimit/TotalCodeSize
     :cognitect.aws.lambda.AccountLimit/ConcurrentExecutions
     :cognitect.aws.lambda.AccountLimit/CodeSizeUnzipped
     :cognitect.aws.lambda.AccountLimit/CodeSizeZipped]))

(s/def :cognitect.aws.lambda/GetAccountSettingsRequest (s/keys))

(s/def
  :cognitect.aws.lambda/ListFunctionsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListFunctionsResponse/Functions
     :cognitect.aws.lambda.ListFunctionsResponse/NextMarker]))

(s/def
  :cognitect.aws.lambda/ListEventSourceMappingsResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.ListEventSourceMappingsResponse/EventSourceMappings
     :cognitect.aws.lambda.ListEventSourceMappingsResponse/NextMarker]))

(s/def
  :cognitect.aws.lambda/Version
  (s/with-gen #(re-matches (re-pattern "(\\$LATEST|[0-9]+)") %) #(gen/string)))

(s/def
  :cognitect.aws.lambda/Weight
  (s/with-gen
    (s/and double? #(<= 0 % 1))
    #(gen/double* {:infinite? false, :NaN? false, :min 0, :max 1})))

(s/def
  :cognitect.aws.lambda/EventSourceMappingConfiguration
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.EventSourceMappingConfiguration/FunctionArn
     :cognitect.aws.lambda.EventSourceMappingConfiguration/BatchSize
     :cognitect.aws.lambda.EventSourceMappingConfiguration/EventSourceArn
     :cognitect.aws.lambda.EventSourceMappingConfiguration/MaximumBatchingWindowInSeconds
     :cognitect.aws.lambda.EventSourceMappingConfiguration/LastProcessingResult
     :cognitect.aws.lambda.EventSourceMappingConfiguration/LastModified
     :cognitect.aws.lambda.EventSourceMappingConfiguration/State
     :cognitect.aws.lambda.EventSourceMappingConfiguration/UUID
     :cognitect.aws.lambda.EventSourceMappingConfiguration/StateTransitionReason]))

(s/def :cognitect.aws.lambda/LayersList (s/coll-of :cognitect.aws.lambda/LayersListItem))

(s/def
  :cognitect.aws.lambda/LayerVersionsList
  (s/coll-of :cognitect.aws.lambda/LayerVersionsListItem))

(s/def
  :cognitect.aws.lambda/AddLayerVersionPermissionResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.AddLayerVersionPermissionResponse/Statement
     :cognitect.aws.lambda.AddLayerVersionPermissionResponse/RevisionId]))

(s/def
  :cognitect.aws.lambda/Layer
  (s/keys :opt-un [:cognitect.aws.lambda.Layer/CodeSize :cognitect.aws.lambda.Layer/Arn]))

(s/def
  :cognitect.aws.lambda/Handler
  (s/with-gen #(re-matches (re-pattern "[^\\s]+") %) #(gen/string)))

(s/def :cognitect.aws.lambda/VpcId string?)

(s/def
  :cognitect.aws.lambda/GetPolicyRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.GetPolicyRequest/FunctionName]
    :opt-un
    [:cognitect.aws.lambda.GetPolicyRequest/Qualifier]))

(s/def
  :cognitect.aws.lambda/RemovePermissionRequest
  (s/keys
    :req-un
    [:cognitect.aws.lambda.RemovePermissionRequest/FunctionName
     :cognitect.aws.lambda.RemovePermissionRequest/StatementId]
    :opt-un
    [:cognitect.aws.lambda.RemovePermissionRequest/RevisionId
     :cognitect.aws.lambda.RemovePermissionRequest/Qualifier]))

(s/def :cognitect.aws.lambda/FunctionList (s/coll-of :cognitect.aws.lambda/FunctionConfiguration))

(s/def :cognitect.aws.lambda/String string?)

(s/def
  :cognitect.aws.lambda/BlobStream
  (s/or :byte-array bytes? :input-stream #(instance? java.io.InputStream %)))

(s/def
  :cognitect.aws.lambda/GetFunctionResponse
  (s/keys
    :opt-un
    [:cognitect.aws.lambda.GetFunctionResponse/Tags
     :cognitect.aws.lambda.GetFunctionResponse/Concurrency
     :cognitect.aws.lambda.GetFunctionResponse/Code
     :cognitect.aws.lambda.GetFunctionResponse/Configuration]))

(s/def
  :cognitect.aws.lambda/TracingConfigResponse
  (s/keys :opt-un [:cognitect.aws.lambda.TracingConfigResponse/Mode]))

(s/def :cognitect.aws.lambda/TagKeyList (s/coll-of :cognitect.aws.lambda/TagKey))

(s/def :cognitect.aws.lambda/SensitiveString string?)

(s/def :cognitect.aws.lambda.GetLayerVersionPolicyRequest/LayerName :cognitect.aws.lambda/LayerName)

(s/def
  :cognitect.aws.lambda.GetLayerVersionPolicyRequest/VersionNumber
  :cognitect.aws.lambda/LayerVersionNumber)

(s/def
  :cognitect.aws.lambda.ListLayerVersionsRequest/CompatibleRuntime
  :cognitect.aws.lambda/Runtime)

(s/def :cognitect.aws.lambda.ListLayerVersionsRequest/LayerName :cognitect.aws.lambda/LayerName)

(s/def :cognitect.aws.lambda.ListLayerVersionsRequest/Marker :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.ListLayerVersionsRequest/MaxItems
  :cognitect.aws.lambda/MaxLayerListItems)

(s/def :cognitect.aws.lambda.TagResourceRequest/Resource :cognitect.aws.lambda/FunctionArn)

(s/def :cognitect.aws.lambda.TagResourceRequest/Tags :cognitect.aws.lambda/Tags)

(s/def
  :cognitect.aws.lambda.RemoveLayerVersionPermissionRequest/LayerName
  :cognitect.aws.lambda/LayerName)

(s/def
  :cognitect.aws.lambda.RemoveLayerVersionPermissionRequest/VersionNumber
  :cognitect.aws.lambda/LayerVersionNumber)

(s/def
  :cognitect.aws.lambda.RemoveLayerVersionPermissionRequest/StatementId
  :cognitect.aws.lambda/StatementId)

(s/def
  :cognitect.aws.lambda.RemoveLayerVersionPermissionRequest/RevisionId
  :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.PublishLayerVersionRequest/LayerName :cognitect.aws.lambda/LayerName)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionRequest/Description
  :cognitect.aws.lambda/Description)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionRequest/Content
  :cognitect.aws.lambda/LayerVersionContentInput)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionRequest/CompatibleRuntimes
  :cognitect.aws.lambda/CompatibleRuntimes)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionRequest/LicenseInfo
  :cognitect.aws.lambda/LicenseInfo)

(s/def :cognitect.aws.lambda.LayerVersionContentInput/S3Bucket :cognitect.aws.lambda/S3Bucket)

(s/def :cognitect.aws.lambda.LayerVersionContentInput/S3Key :cognitect.aws.lambda/S3Key)

(s/def
  :cognitect.aws.lambda.LayerVersionContentInput/S3ObjectVersion
  :cognitect.aws.lambda/S3ObjectVersion)

(s/def :cognitect.aws.lambda.LayerVersionContentInput/ZipFile :cognitect.aws.lambda/Blob)

(s/def :cognitect.aws.lambda.ListTagsRequest/Resource :cognitect.aws.lambda/FunctionArn)

(s/def
  :cognitect.aws.lambda.GetFunctionRequest/FunctionName
  :cognitect.aws.lambda/NamespacedFunctionName)

(s/def :cognitect.aws.lambda.GetFunctionRequest/Qualifier :cognitect.aws.lambda/Qualifier)

(s/def :cognitect.aws.lambda.Environment/Variables :cognitect.aws.lambda/EnvironmentVariables)

(s/def
  :cognitect.aws.lambda.UpdateFunctionCodeRequest/FunctionName
  :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.UpdateFunctionCodeRequest/ZipFile :cognitect.aws.lambda/Blob)

(s/def :cognitect.aws.lambda.UpdateFunctionCodeRequest/S3Bucket :cognitect.aws.lambda/S3Bucket)

(s/def :cognitect.aws.lambda.UpdateFunctionCodeRequest/S3Key :cognitect.aws.lambda/S3Key)

(s/def
  :cognitect.aws.lambda.UpdateFunctionCodeRequest/S3ObjectVersion
  :cognitect.aws.lambda/S3ObjectVersion)

(s/def :cognitect.aws.lambda.UpdateFunctionCodeRequest/Publish :cognitect.aws.lambda/Boolean)

(s/def :cognitect.aws.lambda.UpdateFunctionCodeRequest/DryRun :cognitect.aws.lambda/Boolean)

(s/def :cognitect.aws.lambda.UpdateFunctionCodeRequest/RevisionId :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.GetEventSourceMappingRequest/UUID :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionRequest/LayerName
  :cognitect.aws.lambda/LayerName)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionRequest/VersionNumber
  :cognitect.aws.lambda/LayerVersionNumber)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionRequest/StatementId
  :cognitect.aws.lambda/StatementId)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionRequest/Action
  :cognitect.aws.lambda/LayerPermissionAllowedAction)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionRequest/Principal
  :cognitect.aws.lambda/LayerPermissionAllowedPrincipal)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionRequest/OrganizationId
  :cognitect.aws.lambda/OrganizationId)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionRequest/RevisionId
  :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.InvokeAsyncResponse/Status :cognitect.aws.lambda/HttpStatus)

(s/def :cognitect.aws.lambda.InvocationResponse/StatusCode :cognitect.aws.lambda/Integer)

(s/def :cognitect.aws.lambda.InvocationResponse/FunctionError :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.InvocationResponse/LogResult :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.InvocationResponse/Payload :cognitect.aws.lambda/Blob)

(s/def :cognitect.aws.lambda.InvocationResponse/ExecutedVersion :cognitect.aws.lambda/Version)

(s/def
  :cognitect.aws.lambda.DeleteFunctionConcurrencyRequest/FunctionName
  :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.ListAliasesRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.ListAliasesRequest/FunctionVersion :cognitect.aws.lambda/Version)

(s/def :cognitect.aws.lambda.ListAliasesRequest/Marker :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.ListAliasesRequest/MaxItems :cognitect.aws.lambda/MaxListItems)

(s/def
  :cognitect.aws.lambda.GetFunctionConfigurationRequest/FunctionName
  :cognitect.aws.lambda/NamespacedFunctionName)

(s/def
  :cognitect.aws.lambda.GetFunctionConfigurationRequest/Qualifier
  :cognitect.aws.lambda/Qualifier)

(s/def
  :cognitect.aws.lambda.ListVersionsByFunctionRequest/FunctionName
  :cognitect.aws.lambda/NamespacedFunctionName)

(s/def :cognitect.aws.lambda.ListVersionsByFunctionRequest/Marker :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.ListVersionsByFunctionRequest/MaxItems
  :cognitect.aws.lambda/MaxListItems)

(s/def :cognitect.aws.lambda.LayersListItem/LayerName :cognitect.aws.lambda/LayerName)

(s/def :cognitect.aws.lambda.LayersListItem/LayerArn :cognitect.aws.lambda/LayerArn)

(s/def
  :cognitect.aws.lambda.LayersListItem/LatestMatchingVersion
  :cognitect.aws.lambda/LayerVersionsListItem)

(s/def :cognitect.aws.lambda.VpcConfig/SubnetIds :cognitect.aws.lambda/SubnetIds)

(s/def :cognitect.aws.lambda.VpcConfig/SecurityGroupIds :cognitect.aws.lambda/SecurityGroupIds)

(s/def
  :cognitect.aws.lambda.FunctionConfiguration/Environment
  :cognitect.aws.lambda/EnvironmentResponse)

(s/def :cognitect.aws.lambda.FunctionConfiguration/LastModified :cognitect.aws.lambda/Timestamp)

(s/def :cognitect.aws.lambda.FunctionConfiguration/CodeSha256 :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.FunctionConfiguration/Timeout :cognitect.aws.lambda/Timeout)

(s/def :cognitect.aws.lambda.FunctionConfiguration/KMSKeyArn :cognitect.aws.lambda/KMSKeyArn)

(s/def
  :cognitect.aws.lambda.FunctionConfiguration/VpcConfig
  :cognitect.aws.lambda/VpcConfigResponse)

(s/def
  :cognitect.aws.lambda.FunctionConfiguration/DeadLetterConfig
  :cognitect.aws.lambda/DeadLetterConfig)

(s/def
  :cognitect.aws.lambda.FunctionConfiguration/FunctionName
  :cognitect.aws.lambda/NamespacedFunctionName)

(s/def
  :cognitect.aws.lambda.FunctionConfiguration/TracingConfig
  :cognitect.aws.lambda/TracingConfigResponse)

(s/def
  :cognitect.aws.lambda.FunctionConfiguration/FunctionArn
  :cognitect.aws.lambda/NameSpacedFunctionArn)

(s/def :cognitect.aws.lambda.FunctionConfiguration/MasterArn :cognitect.aws.lambda/FunctionArn)

(s/def :cognitect.aws.lambda.FunctionConfiguration/RevisionId :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.FunctionConfiguration/Description :cognitect.aws.lambda/Description)

(s/def :cognitect.aws.lambda.FunctionConfiguration/CodeSize :cognitect.aws.lambda/Long)

(s/def :cognitect.aws.lambda.FunctionConfiguration/Layers :cognitect.aws.lambda/LayersReferenceList)

(s/def :cognitect.aws.lambda.FunctionConfiguration/Runtime :cognitect.aws.lambda/Runtime)

(s/def :cognitect.aws.lambda.FunctionConfiguration/Role :cognitect.aws.lambda/RoleArn)

(s/def :cognitect.aws.lambda.FunctionConfiguration/MemorySize :cognitect.aws.lambda/MemorySize)

(s/def :cognitect.aws.lambda.FunctionConfiguration/Version :cognitect.aws.lambda/Version)

(s/def :cognitect.aws.lambda.FunctionConfiguration/Handler :cognitect.aws.lambda/Handler)

(s/def
  :cognitect.aws.lambda.GetAccountSettingsResponse/AccountLimit
  :cognitect.aws.lambda/AccountLimit)

(s/def
  :cognitect.aws.lambda.GetAccountSettingsResponse/AccountUsage
  :cognitect.aws.lambda/AccountUsage)

(s/def :cognitect.aws.lambda.GetLayerVersionRequest/LayerName :cognitect.aws.lambda/LayerName)

(s/def
  :cognitect.aws.lambda.GetLayerVersionRequest/VersionNumber
  :cognitect.aws.lambda/LayerVersionNumber)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Environment
  :cognitect.aws.lambda/Environment)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Timeout
  :cognitect.aws.lambda/Timeout)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/KMSKeyArn
  :cognitect.aws.lambda/KMSKeyArn)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/VpcConfig
  :cognitect.aws.lambda/VpcConfig)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/DeadLetterConfig
  :cognitect.aws.lambda/DeadLetterConfig)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/FunctionName
  :cognitect.aws.lambda/FunctionName)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/TracingConfig
  :cognitect.aws.lambda/TracingConfig)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/RevisionId
  :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Description
  :cognitect.aws.lambda/Description)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Layers
  :cognitect.aws.lambda/LayerList)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Runtime
  :cognitect.aws.lambda/Runtime)

(s/def :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Role :cognitect.aws.lambda/RoleArn)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/MemorySize
  :cognitect.aws.lambda/MemorySize)

(s/def
  :cognitect.aws.lambda.UpdateFunctionConfigurationRequest/Handler
  :cognitect.aws.lambda/Handler)

(s/def :cognitect.aws.lambda.ListAliasesResponse/NextMarker :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.ListAliasesResponse/Aliases :cognitect.aws.lambda/AliasList)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionResponse/Content
  :cognitect.aws.lambda/LayerVersionContentOutput)

(s/def :cognitect.aws.lambda.PublishLayerVersionResponse/LayerArn :cognitect.aws.lambda/LayerArn)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionResponse/LayerVersionArn
  :cognitect.aws.lambda/LayerVersionArn)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionResponse/Description
  :cognitect.aws.lambda/Description)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionResponse/CreatedDate
  :cognitect.aws.lambda/Timestamp)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionResponse/Version
  :cognitect.aws.lambda/LayerVersionNumber)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionResponse/CompatibleRuntimes
  :cognitect.aws.lambda/CompatibleRuntimes)

(s/def
  :cognitect.aws.lambda.PublishLayerVersionResponse/LicenseInfo
  :cognitect.aws.lambda/LicenseInfo)

(s/def :cognitect.aws.lambda.PublishVersionRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.PublishVersionRequest/CodeSha256 :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.PublishVersionRequest/Description :cognitect.aws.lambda/Description)

(s/def :cognitect.aws.lambda.PublishVersionRequest/RevisionId :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.UpdateAliasRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.UpdateAliasRequest/Name :cognitect.aws.lambda/Alias)

(s/def :cognitect.aws.lambda.UpdateAliasRequest/FunctionVersion :cognitect.aws.lambda/Version)

(s/def :cognitect.aws.lambda.UpdateAliasRequest/Description :cognitect.aws.lambda/Description)

(s/def
  :cognitect.aws.lambda.UpdateAliasRequest/RoutingConfig
  :cognitect.aws.lambda/AliasRoutingConfiguration)

(s/def :cognitect.aws.lambda.UpdateAliasRequest/RevisionId :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.FunctionCode/ZipFile :cognitect.aws.lambda/Blob)

(s/def :cognitect.aws.lambda.FunctionCode/S3Bucket :cognitect.aws.lambda/S3Bucket)

(s/def :cognitect.aws.lambda.FunctionCode/S3Key :cognitect.aws.lambda/S3Key)

(s/def :cognitect.aws.lambda.FunctionCode/S3ObjectVersion :cognitect.aws.lambda/S3ObjectVersion)

(s/def :cognitect.aws.lambda.DeadLetterConfig/TargetArn :cognitect.aws.lambda/ResourceArn)

(s/def :cognitect.aws.lambda.AliasConfiguration/AliasArn :cognitect.aws.lambda/FunctionArn)

(s/def :cognitect.aws.lambda.AliasConfiguration/Name :cognitect.aws.lambda/Alias)

(s/def :cognitect.aws.lambda.AliasConfiguration/FunctionVersion :cognitect.aws.lambda/Version)

(s/def :cognitect.aws.lambda.AliasConfiguration/Description :cognitect.aws.lambda/Description)

(s/def
  :cognitect.aws.lambda.AliasConfiguration/RoutingConfig
  :cognitect.aws.lambda/AliasRoutingConfiguration)

(s/def :cognitect.aws.lambda.AliasConfiguration/RevisionId :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.ListLayersResponse/NextMarker :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.ListLayersResponse/Layers :cognitect.aws.lambda/LayersList)

(s/def :cognitect.aws.lambda.GetLayerVersionByArnRequest/Arn :cognitect.aws.lambda/LayerVersionArn)

(s/def :cognitect.aws.lambda.ListLayersRequest/CompatibleRuntime :cognitect.aws.lambda/Runtime)

(s/def :cognitect.aws.lambda.ListLayersRequest/Marker :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.ListLayersRequest/MaxItems :cognitect.aws.lambda/MaxLayerListItems)

(s/def :cognitect.aws.lambda.DeleteEventSourceMappingRequest/UUID :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.EnvironmentError/ErrorCode :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.EnvironmentError/Message :cognitect.aws.lambda/SensitiveString)

(s/def :cognitect.aws.lambda.ListTagsResponse/Tags :cognitect.aws.lambda/Tags)

(s/def
  :cognitect.aws.lambda.InvocationRequest/FunctionName
  :cognitect.aws.lambda/NamespacedFunctionName)

(s/def :cognitect.aws.lambda.InvocationRequest/InvocationType :cognitect.aws.lambda/InvocationType)

(s/def :cognitect.aws.lambda.InvocationRequest/LogType :cognitect.aws.lambda/LogType)

(s/def :cognitect.aws.lambda.InvocationRequest/ClientContext :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.InvocationRequest/Payload :cognitect.aws.lambda/Blob)

(s/def :cognitect.aws.lambda.InvocationRequest/Qualifier :cognitect.aws.lambda/Qualifier)

(s/def :cognitect.aws.lambda.AccountUsage/TotalCodeSize :cognitect.aws.lambda/Long)

(s/def :cognitect.aws.lambda.AccountUsage/FunctionCount :cognitect.aws.lambda/Long)

(s/def :cognitect.aws.lambda.ListFunctionsRequest/MasterRegion :cognitect.aws.lambda/MasterRegion)

(s/def
  :cognitect.aws.lambda.ListFunctionsRequest/FunctionVersion
  :cognitect.aws.lambda/FunctionVersion)

(s/def :cognitect.aws.lambda.ListFunctionsRequest/Marker :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.ListFunctionsRequest/MaxItems :cognitect.aws.lambda/MaxListItems)

(s/def :cognitect.aws.lambda.ListLayerVersionsResponse/NextMarker :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.ListLayerVersionsResponse/LayerVersions
  :cognitect.aws.lambda/LayerVersionsList)

(s/def :cognitect.aws.lambda.TracingConfig/Mode :cognitect.aws.lambda/TracingMode)

(s/def
  :cognitect.aws.lambda.LayerVersionsListItem/LayerVersionArn
  :cognitect.aws.lambda/LayerVersionArn)

(s/def :cognitect.aws.lambda.LayerVersionsListItem/Version :cognitect.aws.lambda/LayerVersionNumber)

(s/def :cognitect.aws.lambda.LayerVersionsListItem/Description :cognitect.aws.lambda/Description)

(s/def :cognitect.aws.lambda.LayerVersionsListItem/CreatedDate :cognitect.aws.lambda/Timestamp)

(s/def
  :cognitect.aws.lambda.LayerVersionsListItem/CompatibleRuntimes
  :cognitect.aws.lambda/CompatibleRuntimes)

(s/def :cognitect.aws.lambda.LayerVersionsListItem/LicenseInfo :cognitect.aws.lambda/LicenseInfo)

(s/def :cognitect.aws.lambda.GetLayerVersionPolicyResponse/Policy :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.GetLayerVersionPolicyResponse/RevisionId :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.CreateAliasRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.CreateAliasRequest/Name :cognitect.aws.lambda/Alias)

(s/def :cognitect.aws.lambda.CreateAliasRequest/FunctionVersion :cognitect.aws.lambda/Version)

(s/def :cognitect.aws.lambda.CreateAliasRequest/Description :cognitect.aws.lambda/Description)

(s/def
  :cognitect.aws.lambda.CreateAliasRequest/RoutingConfig
  :cognitect.aws.lambda/AliasRoutingConfiguration)

(s/def :cognitect.aws.lambda.ListVersionsByFunctionResponse/NextMarker :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.ListVersionsByFunctionResponse/Versions
  :cognitect.aws.lambda/FunctionList)

(s/def
  :cognitect.aws.lambda.ListEventSourceMappingsRequest/EventSourceArn
  :cognitect.aws.lambda/Arn)

(s/def
  :cognitect.aws.lambda.ListEventSourceMappingsRequest/FunctionName
  :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.ListEventSourceMappingsRequest/Marker :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.ListEventSourceMappingsRequest/MaxItems
  :cognitect.aws.lambda/MaxListItems)

(s/def :cognitect.aws.lambda.DeleteFunctionRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.DeleteFunctionRequest/Qualifier :cognitect.aws.lambda/Qualifier)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Environment :cognitect.aws.lambda/Environment)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Timeout :cognitect.aws.lambda/Timeout)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/KMSKeyArn :cognitect.aws.lambda/KMSKeyArn)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/VpcConfig :cognitect.aws.lambda/VpcConfig)

(s/def
  :cognitect.aws.lambda.CreateFunctionRequest/DeadLetterConfig
  :cognitect.aws.lambda/DeadLetterConfig)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Tags :cognitect.aws.lambda/Tags)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Publish :cognitect.aws.lambda/Boolean)

(s/def
  :cognitect.aws.lambda.CreateFunctionRequest/TracingConfig
  :cognitect.aws.lambda/TracingConfig)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Code :cognitect.aws.lambda/FunctionCode)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Description :cognitect.aws.lambda/Description)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Layers :cognitect.aws.lambda/LayerList)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Runtime :cognitect.aws.lambda/Runtime)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Role :cognitect.aws.lambda/RoleArn)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/MemorySize :cognitect.aws.lambda/MemorySize)

(s/def :cognitect.aws.lambda.CreateFunctionRequest/Handler :cognitect.aws.lambda/Handler)

(s/def :cognitect.aws.lambda.VpcConfigResponse/SubnetIds :cognitect.aws.lambda/SubnetIds)

(s/def
  :cognitect.aws.lambda.VpcConfigResponse/SecurityGroupIds
  :cognitect.aws.lambda/SecurityGroupIds)

(s/def :cognitect.aws.lambda.VpcConfigResponse/VpcId :cognitect.aws.lambda/VpcId)

(s/def :cognitect.aws.lambda.AddPermissionResponse/Statement :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.CreateEventSourceMappingRequest/EventSourceArn
  :cognitect.aws.lambda/Arn)

(s/def
  :cognitect.aws.lambda.CreateEventSourceMappingRequest/FunctionName
  :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.CreateEventSourceMappingRequest/Enabled :cognitect.aws.lambda/Enabled)

(s/def
  :cognitect.aws.lambda.CreateEventSourceMappingRequest/BatchSize
  :cognitect.aws.lambda/BatchSize)

(s/def
  :cognitect.aws.lambda.CreateEventSourceMappingRequest/MaximumBatchingWindowInSeconds
  :cognitect.aws.lambda/MaximumBatchingWindowInSeconds)

(s/def
  :cognitect.aws.lambda.CreateEventSourceMappingRequest/StartingPosition
  :cognitect.aws.lambda/EventSourcePosition)

(s/def
  :cognitect.aws.lambda.CreateEventSourceMappingRequest/StartingPositionTimestamp
  :cognitect.aws.lambda/Date)

(s/def :cognitect.aws.lambda.AddPermissionRequest/Principal :cognitect.aws.lambda/Principal)

(s/def :cognitect.aws.lambda.AddPermissionRequest/StatementId :cognitect.aws.lambda/StatementId)

(s/def
  :cognitect.aws.lambda.AddPermissionRequest/EventSourceToken
  :cognitect.aws.lambda/EventSourceToken)

(s/def :cognitect.aws.lambda.AddPermissionRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.AddPermissionRequest/Action :cognitect.aws.lambda/Action)

(s/def :cognitect.aws.lambda.AddPermissionRequest/RevisionId :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.AddPermissionRequest/Qualifier :cognitect.aws.lambda/Qualifier)

(s/def :cognitect.aws.lambda.AddPermissionRequest/SourceAccount :cognitect.aws.lambda/SourceOwner)

(s/def :cognitect.aws.lambda.AddPermissionRequest/SourceArn :cognitect.aws.lambda/Arn)

(s/def
  :cognitect.aws.lambda.PutFunctionConcurrencyRequest/FunctionName
  :cognitect.aws.lambda/FunctionName)

(s/def
  :cognitect.aws.lambda.PutFunctionConcurrencyRequest/ReservedConcurrentExecutions
  :cognitect.aws.lambda/ReservedConcurrentExecutions)

(s/def :cognitect.aws.lambda.FunctionCodeLocation/RepositoryType :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.FunctionCodeLocation/Location :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.GetPolicyResponse/Policy :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.GetPolicyResponse/RevisionId :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.UpdateEventSourceMappingRequest/UUID :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.UpdateEventSourceMappingRequest/FunctionName
  :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.UpdateEventSourceMappingRequest/Enabled :cognitect.aws.lambda/Enabled)

(s/def
  :cognitect.aws.lambda.UpdateEventSourceMappingRequest/BatchSize
  :cognitect.aws.lambda/BatchSize)

(s/def
  :cognitect.aws.lambda.UpdateEventSourceMappingRequest/MaximumBatchingWindowInSeconds
  :cognitect.aws.lambda/MaximumBatchingWindowInSeconds)

(s/def :cognitect.aws.lambda.DeleteLayerVersionRequest/LayerName :cognitect.aws.lambda/LayerName)

(s/def
  :cognitect.aws.lambda.DeleteLayerVersionRequest/VersionNumber
  :cognitect.aws.lambda/LayerVersionNumber)

(s/def :cognitect.aws.lambda.UntagResourceRequest/Resource :cognitect.aws.lambda/FunctionArn)

(s/def :cognitect.aws.lambda.UntagResourceRequest/TagKeys :cognitect.aws.lambda/TagKeyList)

(s/def :cognitect.aws.lambda.DeleteAliasRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.DeleteAliasRequest/Name :cognitect.aws.lambda/Alias)

(s/def
  :cognitect.aws.lambda.Concurrency/ReservedConcurrentExecutions
  :cognitect.aws.lambda/ReservedConcurrentExecutions)

(s/def
  :cognitect.aws.lambda.InvokeAsyncRequest/FunctionName
  :cognitect.aws.lambda/NamespacedFunctionName)

(s/def :cognitect.aws.lambda.InvokeAsyncRequest/InvokeArgs :cognitect.aws.lambda/BlobStream)

(s/def :cognitect.aws.lambda.GetAliasRequest/FunctionName :cognitect.aws.lambda/FunctionName)

(s/def :cognitect.aws.lambda.GetAliasRequest/Name :cognitect.aws.lambda/Alias)

(s/def :cognitect.aws.lambda.LayerVersionContentOutput/Location :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.LayerVersionContentOutput/CodeSha256 :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.LayerVersionContentOutput/CodeSize :cognitect.aws.lambda/Long)

(s/def
  :cognitect.aws.lambda.GetLayerVersionResponse/Content
  :cognitect.aws.lambda/LayerVersionContentOutput)

(s/def :cognitect.aws.lambda.GetLayerVersionResponse/LayerArn :cognitect.aws.lambda/LayerArn)

(s/def
  :cognitect.aws.lambda.GetLayerVersionResponse/LayerVersionArn
  :cognitect.aws.lambda/LayerVersionArn)

(s/def :cognitect.aws.lambda.GetLayerVersionResponse/Description :cognitect.aws.lambda/Description)

(s/def :cognitect.aws.lambda.GetLayerVersionResponse/CreatedDate :cognitect.aws.lambda/Timestamp)

(s/def
  :cognitect.aws.lambda.GetLayerVersionResponse/Version
  :cognitect.aws.lambda/LayerVersionNumber)

(s/def
  :cognitect.aws.lambda.GetLayerVersionResponse/CompatibleRuntimes
  :cognitect.aws.lambda/CompatibleRuntimes)

(s/def :cognitect.aws.lambda.GetLayerVersionResponse/LicenseInfo :cognitect.aws.lambda/LicenseInfo)

(s/def
  :cognitect.aws.lambda.EnvironmentResponse/Variables
  :cognitect.aws.lambda/EnvironmentVariables)

(s/def :cognitect.aws.lambda.EnvironmentResponse/Error :cognitect.aws.lambda/EnvironmentError)

(s/def
  :cognitect.aws.lambda.AliasRoutingConfiguration/AdditionalVersionWeights
  :cognitect.aws.lambda/AdditionalVersionWeights)

(s/def :cognitect.aws.lambda.AccountLimit/TotalCodeSize :cognitect.aws.lambda/Long)

(s/def :cognitect.aws.lambda.AccountLimit/CodeSizeUnzipped :cognitect.aws.lambda/Long)

(s/def :cognitect.aws.lambda.AccountLimit/CodeSizeZipped :cognitect.aws.lambda/Long)

(s/def :cognitect.aws.lambda.AccountLimit/ConcurrentExecutions :cognitect.aws.lambda/Integer)

(s/def
  :cognitect.aws.lambda.AccountLimit/UnreservedConcurrentExecutions
  :cognitect.aws.lambda/UnreservedConcurrentExecutions)

(s/def :cognitect.aws.lambda.ListFunctionsResponse/NextMarker :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.ListFunctionsResponse/Functions :cognitect.aws.lambda/FunctionList)

(s/def
  :cognitect.aws.lambda.ListEventSourceMappingsResponse/NextMarker
  :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.ListEventSourceMappingsResponse/EventSourceMappings
  :cognitect.aws.lambda/EventSourceMappingsList)

(s/def
  :cognitect.aws.lambda.EventSourceMappingConfiguration/BatchSize
  :cognitect.aws.lambda/BatchSize)

(s/def
  :cognitect.aws.lambda.EventSourceMappingConfiguration/StateTransitionReason
  :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.EventSourceMappingConfiguration/LastModified
  :cognitect.aws.lambda/Date)

(s/def
  :cognitect.aws.lambda.EventSourceMappingConfiguration/EventSourceArn
  :cognitect.aws.lambda/Arn)

(s/def
  :cognitect.aws.lambda.EventSourceMappingConfiguration/MaximumBatchingWindowInSeconds
  :cognitect.aws.lambda/MaximumBatchingWindowInSeconds)

(s/def
  :cognitect.aws.lambda.EventSourceMappingConfiguration/FunctionArn
  :cognitect.aws.lambda/FunctionArn)

(s/def
  :cognitect.aws.lambda.EventSourceMappingConfiguration/LastProcessingResult
  :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.EventSourceMappingConfiguration/State :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.EventSourceMappingConfiguration/UUID :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionResponse/Statement
  :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.AddLayerVersionPermissionResponse/RevisionId
  :cognitect.aws.lambda/String)

(s/def :cognitect.aws.lambda.Layer/Arn :cognitect.aws.lambda/LayerVersionArn)

(s/def :cognitect.aws.lambda.Layer/CodeSize :cognitect.aws.lambda/Long)

(s/def
  :cognitect.aws.lambda.GetPolicyRequest/FunctionName
  :cognitect.aws.lambda/NamespacedFunctionName)

(s/def :cognitect.aws.lambda.GetPolicyRequest/Qualifier :cognitect.aws.lambda/Qualifier)

(s/def
  :cognitect.aws.lambda.RemovePermissionRequest/FunctionName
  :cognitect.aws.lambda/FunctionName)

(s/def
  :cognitect.aws.lambda.RemovePermissionRequest/StatementId
  :cognitect.aws.lambda/NamespacedStatementId)

(s/def :cognitect.aws.lambda.RemovePermissionRequest/Qualifier :cognitect.aws.lambda/Qualifier)

(s/def :cognitect.aws.lambda.RemovePermissionRequest/RevisionId :cognitect.aws.lambda/String)

(s/def
  :cognitect.aws.lambda.GetFunctionResponse/Configuration
  :cognitect.aws.lambda/FunctionConfiguration)

(s/def :cognitect.aws.lambda.GetFunctionResponse/Code :cognitect.aws.lambda/FunctionCodeLocation)

(s/def :cognitect.aws.lambda.GetFunctionResponse/Tags :cognitect.aws.lambda/Tags)

(s/def :cognitect.aws.lambda.GetFunctionResponse/Concurrency :cognitect.aws.lambda/Concurrency)

(s/def :cognitect.aws.lambda.TracingConfigResponse/Mode :cognitect.aws.lambda/TracingMode)

