/*
 * Decompiled with CFR 0.152.
 */
package com.configcat;

import com.configcat.AutoPollingMode;
import com.configcat.ConfigCache;
import com.configcat.ConfigCatHooks;
import com.configcat.ConfigCatLogMessages;
import com.configcat.ConfigCatLogger;
import com.configcat.ConfigFetcher;
import com.configcat.Entry;
import com.configcat.FetchResponse;
import com.configcat.LazyLoadingMode;
import com.configcat.PollingMode;
import com.configcat.RefreshResult;
import com.configcat.Result;
import com.configcat.SettingResult;
import java.io.Closeable;
import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;

public class ConfigService
implements Closeable {
    private static final String CACHE_BASE = "%s_config_v5.json_v2";
    private Entry cachedEntry = Entry.EMPTY;
    private String cachedEntryString = "";
    private final ConfigCache cache;
    private final String cacheKey;
    private final ConfigFetcher configFetcher;
    private final ConfigCatLogger logger;
    private final PollingMode pollingMode;
    private ScheduledExecutorService pollScheduler;
    private ScheduledExecutorService initScheduler;
    private CompletableFuture<Result<Entry>> runningTask;
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final AtomicBoolean offline;
    private final ReentrantLock lock = new ReentrantLock(true);
    private final ConfigCatHooks configCatHooks;

    public ConfigService(String sdkKey, ConfigFetcher configFetcher, PollingMode pollingMode, ConfigCache cache, ConfigCatLogger logger, boolean offline, ConfigCatHooks configCatHooks) {
        this.configFetcher = configFetcher;
        this.pollingMode = pollingMode;
        this.cacheKey = new String(Hex.encodeHex((byte[])DigestUtils.sha1((String)String.format(CACHE_BASE, sdkKey))));
        this.cache = cache;
        this.logger = logger;
        this.offline = new AtomicBoolean(offline);
        this.configCatHooks = configCatHooks;
        if (pollingMode instanceof AutoPollingMode && !offline) {
            AutoPollingMode autoPollingMode = (AutoPollingMode)pollingMode;
            this.startPoll(autoPollingMode);
            this.initScheduler = Executors.newSingleThreadScheduledExecutor();
            this.initScheduler.schedule(() -> {
                this.lock.lock();
                try {
                    if (this.initialized.compareAndSet(false, true)) {
                        this.configCatHooks.invokeOnClientReady();
                        String message = ConfigCatLogMessages.getAutoPollMaxInitWaitTimeReached(autoPollingMode.getMaxInitWaitTimeSeconds());
                        this.logger.warn(4200, message);
                        this.completeRunningTask(Result.error(message, this.cachedEntry));
                    }
                }
                finally {
                    this.lock.unlock();
                }
            }, (long)autoPollingMode.getMaxInitWaitTimeSeconds(), TimeUnit.SECONDS);
        } else {
            this.setInitialized();
        }
    }

    private void setInitialized() {
        if (this.initialized.compareAndSet(false, true)) {
            this.configCatHooks.invokeOnClientReady();
        }
    }

    private void startPoll(AutoPollingMode autoPollingMode) {
        long ageThreshold = (long)((double)((long)autoPollingMode.getAutoPollRateInSeconds() * 1000L) * 0.7);
        this.pollScheduler = Executors.newSingleThreadScheduledExecutor();
        this.pollScheduler.scheduleAtFixedRate(() -> this.fetchIfOlder(System.currentTimeMillis() - ageThreshold, false), 0L, autoPollingMode.getAutoPollRateInSeconds(), TimeUnit.SECONDS);
    }

    public CompletableFuture<RefreshResult> refresh() {
        if (this.offline.get()) {
            String offlineWarning = "Client is in offline mode, it cannot initiate HTTP calls.";
            this.logger.warn(3200, offlineWarning);
            return CompletableFuture.completedFuture(new RefreshResult(false, offlineWarning));
        }
        return this.fetchIfOlder(Long.MAX_VALUE, false).thenApply(entryResult -> new RefreshResult(entryResult.error() == null, entryResult.error()));
    }

    public CompletableFuture<SettingResult> getSettings() {
        if (this.pollingMode instanceof LazyLoadingMode) {
            LazyLoadingMode lazyLoadingMode = (LazyLoadingMode)this.pollingMode;
            return this.fetchIfOlder(System.currentTimeMillis() - (long)lazyLoadingMode.getCacheRefreshIntervalInSeconds() * 1000L, false).thenApply(entryResult -> !((Entry)entryResult.value()).isEmpty() ? new SettingResult(((Entry)entryResult.value()).getConfig().getEntries(), ((Entry)entryResult.value()).getFetchTime()) : SettingResult.EMPTY);
        }
        return this.fetchIfOlder(0L, this.initialized.get()).thenApply(entryResult -> !((Entry)entryResult.value()).isEmpty() ? new SettingResult(((Entry)entryResult.value()).getConfig().getEntries(), ((Entry)entryResult.value()).getFetchTime()) : SettingResult.EMPTY);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<Result<Entry>> fetchIfOlder(long threshold, boolean preferCached) {
        this.lock.lock();
        try {
            Entry fromCache = this.readCache();
            if (!fromCache.isEmpty() && !fromCache.getETag().equals(this.cachedEntry.getETag())) {
                this.configCatHooks.invokeOnConfigChanged(fromCache.getConfig().getEntries());
                this.cachedEntry = fromCache;
            }
            if (this.cachedEntry.getFetchTime() > threshold) {
                this.setInitialized();
                CompletableFuture<Result<Entry>> completableFuture = CompletableFuture.completedFuture(Result.success(this.cachedEntry));
                return completableFuture;
            }
            if (this.offline.get() || preferCached) {
                CompletableFuture<Result<Entry>> completableFuture = CompletableFuture.completedFuture(Result.success(this.cachedEntry));
                return completableFuture;
            }
            if (this.runningTask == null) {
                this.runningTask = new CompletableFuture();
                this.configFetcher.fetchAsync(this.cachedEntry.getETag()).thenAccept(this::processResponse);
            }
            CompletableFuture<Result<Entry>> completableFuture = this.runningTask;
            return completableFuture;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.closed.compareAndSet(false, true)) {
            return;
        }
        if (this.pollingMode instanceof AutoPollingMode) {
            if (this.pollScheduler != null) {
                this.pollScheduler.shutdown();
            }
            if (this.initScheduler != null) {
                this.initScheduler.shutdown();
            }
        }
        this.configFetcher.close();
    }

    public void setOnline() {
        this.lock.lock();
        try {
            if (!this.offline.compareAndSet(true, false)) {
                return;
            }
            if (this.pollingMode instanceof AutoPollingMode) {
                this.startPoll((AutoPollingMode)this.pollingMode);
            }
            this.logger.info(5200, ConfigCatLogMessages.getConfigServiceStatusChanged("ONLINE"));
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setOffline() {
        this.lock.lock();
        try {
            if (!this.offline.compareAndSet(false, true)) {
                return;
            }
            if (this.pollScheduler != null) {
                this.pollScheduler.shutdown();
            }
            if (this.initScheduler != null) {
                this.initScheduler.shutdown();
            }
            this.logger.info(5200, ConfigCatLogMessages.getConfigServiceStatusChanged("OFFLINE"));
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isOffline() {
        return this.offline.get();
    }

    private void processResponse(FetchResponse response) {
        this.lock.lock();
        try {
            this.setInitialized();
            if (response.isFetched()) {
                Entry entry;
                this.cachedEntry = entry = response.entry();
                this.writeCache(entry);
                this.configCatHooks.invokeOnConfigChanged(entry.getConfig().getEntries());
                this.completeRunningTask(Result.success(entry));
            } else {
                if (response.isFetchTimeUpdatable()) {
                    this.cachedEntry = this.cachedEntry.withFetchTime(System.currentTimeMillis());
                    this.writeCache(this.cachedEntry);
                }
                this.completeRunningTask(response.isFailed() ? Result.error(response.error(), this.cachedEntry) : Result.success(this.cachedEntry));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void completeRunningTask(Result<Entry> result) {
        this.runningTask.complete(result);
        this.runningTask = null;
    }

    private Entry readCache() {
        try {
            String cachedConfigJson = this.cache.read(this.cacheKey);
            if (cachedConfigJson != null && cachedConfigJson.equals(this.cachedEntryString)) {
                return Entry.EMPTY;
            }
            this.cachedEntryString = cachedConfigJson;
            Entry deserialized = Entry.fromString(cachedConfigJson);
            return deserialized == null || deserialized.getConfig() == null ? Entry.EMPTY : deserialized;
        }
        catch (Exception e) {
            this.logger.error(2200, "Error occurred while reading the cache.", e);
            return Entry.EMPTY;
        }
    }

    private void writeCache(Entry entry) {
        try {
            String configToCache;
            this.cachedEntryString = configToCache = entry.serialize();
            this.cache.write(this.cacheKey, configToCache);
        }
        catch (Exception e) {
            this.logger.error(2201, "Error occurred while writing the cache", e);
        }
    }
}

