/*
 * Decompiled with CFR 0.152.
 */
package com.configcat;

import com.configcat.AutoPollingMode;
import com.configcat.ClientCacheState;
import com.configcat.ConfigCache;
import com.configcat.ConfigCatHooks;
import com.configcat.ConfigCatLogMessages;
import com.configcat.ConfigCatLogger;
import com.configcat.ConfigFetcher;
import com.configcat.Entry;
import com.configcat.FetchResponse;
import com.configcat.FormattableLogMessage;
import com.configcat.LazyLoadingMode;
import com.configcat.ManualPollingMode;
import com.configcat.PollingMode;
import com.configcat.RefreshResult;
import com.configcat.Result;
import com.configcat.SettingResult;
import com.configcat.Utils;
import java.io.Closeable;
import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;

public class ConfigService
implements Closeable {
    private static final String CACHE_BASE = "%s_config_v6.json_v2";
    private final AtomicReference<Entry> cachedEntry = new AtomicReference<Entry>(Entry.EMPTY);
    private final ConfigCache cache;
    private final String cacheKey;
    private final ConfigFetcher configFetcher;
    private final ConfigCatLogger logger;
    private final PollingMode pollingMode;
    private ScheduledExecutorService pollScheduler;
    private ScheduledExecutorService initScheduler;
    private CompletableFuture<Result<Entry>> runningTask;
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final AtomicBoolean offline;
    private final ReentrantLock lock = new ReentrantLock(true);
    private final ConfigCatHooks configCatHooks;

    public ConfigService(String sdkKey, ConfigFetcher configFetcher, PollingMode pollingMode, ConfigCache cache, ConfigCatLogger logger, boolean offline, ConfigCatHooks configCatHooks) {
        this.configFetcher = configFetcher;
        this.pollingMode = pollingMode;
        this.cacheKey = Utils.sha1(String.format(CACHE_BASE, sdkKey));
        this.cache = cache;
        this.logger = logger;
        this.offline = new AtomicBoolean(offline);
        this.configCatHooks = configCatHooks;
        if (pollingMode instanceof AutoPollingMode && !offline) {
            AutoPollingMode autoPollingMode = (AutoPollingMode)pollingMode;
            this.startPoll(autoPollingMode);
            this.initScheduler = Executors.newSingleThreadScheduledExecutor();
            this.initScheduler.schedule(() -> {
                if (this.initialized.compareAndSet(false, true)) {
                    this.lock.lock();
                    try {
                        this.configCatHooks.invokeOnClientReady(this.determineCacheState(this.cachedEntry.get()));
                        FormattableLogMessage formattableLogMessage = ConfigCatLogMessages.getAutoPollMaxInitWaitTimeReached(autoPollingMode.getMaxInitWaitTimeSeconds());
                        this.logger.warn(4200, formattableLogMessage);
                        this.completeRunningTask(Result.error(formattableLogMessage, this.cachedEntry.get()));
                    }
                    finally {
                        this.lock.unlock();
                    }
                }
            }, (long)autoPollingMode.getMaxInitWaitTimeSeconds(), TimeUnit.SECONDS);
        } else {
            this.cachedEntry.set(this.readCache());
            this.setInitialized();
        }
    }

    private void setInitialized() {
        if (this.initialized.compareAndSet(false, true)) {
            this.configCatHooks.invokeOnClientReady(this.determineCacheState(this.cachedEntry.get()));
        }
    }

    private void startPoll(AutoPollingMode autoPollingMode) {
        long ageThreshold = (long)autoPollingMode.getAutoPollRateInSeconds() * 1000L - 500L;
        this.pollScheduler = Executors.newSingleThreadScheduledExecutor();
        this.pollScheduler.scheduleAtFixedRate(() -> this.fetchIfOlder(System.currentTimeMillis() - ageThreshold, false), 0L, autoPollingMode.getAutoPollRateInSeconds(), TimeUnit.SECONDS);
    }

    public CompletableFuture<RefreshResult> refresh() {
        if (this.offline.get()) {
            String offlineWarning = "Client is in offline mode, it cannot initiate HTTP calls.";
            this.logger.warn(3200, offlineWarning);
            return CompletableFuture.completedFuture(new RefreshResult(false, offlineWarning));
        }
        return this.fetchIfOlder(Long.MAX_VALUE, false).thenApply(entryResult -> new RefreshResult(entryResult.error() == null, entryResult.error()));
    }

    public CompletableFuture<SettingResult> getSettings() {
        if (this.pollingMode instanceof LazyLoadingMode) {
            LazyLoadingMode lazyLoadingMode = (LazyLoadingMode)this.pollingMode;
            return this.fetchIfOlder(System.currentTimeMillis() - (long)lazyLoadingMode.getCacheRefreshIntervalInSeconds() * 1000L, false).thenApply(entryResult -> !((Entry)entryResult.value()).isEmpty() ? new SettingResult(((Entry)entryResult.value()).getConfig().getEntries(), ((Entry)entryResult.value()).getFetchTime()) : SettingResult.EMPTY);
        }
        long threshold = 0L;
        if (!this.initialized.get() && this.pollingMode instanceof AutoPollingMode) {
            AutoPollingMode autoPollingMode = (AutoPollingMode)this.pollingMode;
            threshold = System.currentTimeMillis() - (long)autoPollingMode.getAutoPollRateInSeconds() * 1000L;
        }
        return this.fetchIfOlder(threshold, this.initialized.get()).thenApply(entryResult -> !((Entry)entryResult.value()).isEmpty() ? new SettingResult(((Entry)entryResult.value()).getConfig().getEntries(), ((Entry)entryResult.value()).getFetchTime()) : SettingResult.EMPTY);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<Result<Entry>> fetchIfOlder(long threshold, boolean preferCached) {
        Entry fromCache = this.readCache();
        if (!fromCache.isEmpty() && !fromCache.getETag().equals(this.cachedEntry.get().getETag()) && fromCache.getFetchTime() > this.cachedEntry.get().getFetchTime()) {
            this.configCatHooks.invokeOnConfigChanged(fromCache.getConfig().getEntries());
            this.cachedEntry.set(fromCache);
        }
        if (this.cachedEntry.get().getFetchTime() > threshold) {
            this.setInitialized();
            return CompletableFuture.completedFuture(Result.success(this.cachedEntry.get()));
        }
        if (this.offline.get() || preferCached) {
            return CompletableFuture.completedFuture(Result.success(this.cachedEntry.get()));
        }
        this.lock.lock();
        try {
            if (this.runningTask == null) {
                this.runningTask = new CompletableFuture();
                this.configFetcher.fetchAsync(this.cachedEntry.get().getETag()).thenAccept(this::processResponse);
            }
            CompletableFuture<Result<Entry>> completableFuture = this.runningTask;
            return completableFuture;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.closed.compareAndSet(false, true)) {
            return;
        }
        if (this.pollingMode instanceof AutoPollingMode) {
            if (this.pollScheduler != null) {
                this.pollScheduler.shutdown();
            }
            if (this.initScheduler != null) {
                this.initScheduler.shutdown();
            }
        }
        this.configFetcher.close();
    }

    public void setOnline() {
        this.lock.lock();
        try {
            if (!this.offline.compareAndSet(true, false)) {
                return;
            }
            if (this.pollingMode instanceof AutoPollingMode) {
                this.startPoll((AutoPollingMode)this.pollingMode);
            }
            this.logger.info(5200, ConfigCatLogMessages.getConfigServiceStatusChanged("ONLINE"));
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setOffline() {
        this.lock.lock();
        try {
            if (!this.offline.compareAndSet(false, true)) {
                return;
            }
            if (this.pollScheduler != null) {
                this.pollScheduler.shutdown();
            }
            if (this.initScheduler != null) {
                this.initScheduler.shutdown();
            }
            this.logger.info(5200, ConfigCatLogMessages.getConfigServiceStatusChanged("OFFLINE"));
        }
        finally {
            this.lock.unlock();
        }
    }

    public boolean isOffline() {
        return this.offline.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processResponse(FetchResponse response) {
        Entry previousEntry = this.cachedEntry.get();
        this.lock.lock();
        try {
            if (response.isFetched()) {
                Entry entry = response.entry();
                this.cachedEntry.set(entry);
                this.writeCache(entry);
                this.configCatHooks.invokeOnConfigChanged(entry.getConfig().getEntries());
                this.completeRunningTask(Result.success(entry));
            } else {
                if (response.isFetchTimeUpdatable()) {
                    this.cachedEntry.set(previousEntry.withFetchTime(System.currentTimeMillis()));
                    this.writeCache(this.cachedEntry.get());
                }
                this.completeRunningTask(response.isFailed() ? Result.error(response.error(), this.cachedEntry.get()) : Result.success(this.cachedEntry.get()));
            }
            this.setInitialized();
        }
        finally {
            this.lock.unlock();
        }
    }

    private void completeRunningTask(Result<Entry> result) {
        this.runningTask.complete(result);
        this.runningTask = null;
    }

    private Entry readCache() {
        try {
            String cachedConfigJson = this.cache.read(this.cacheKey);
            if (cachedConfigJson != null && cachedConfigJson.equals(this.cachedEntry.get().getCacheString())) {
                return Entry.EMPTY;
            }
            Entry deserialized = Entry.fromString(cachedConfigJson);
            return deserialized == null || deserialized.getConfig() == null ? Entry.EMPTY : deserialized;
        }
        catch (Exception e) {
            this.logger.error(2200, "Error occurred while reading the cache.", e);
            return Entry.EMPTY;
        }
    }

    private void writeCache(Entry entry) {
        try {
            this.cache.write(this.cacheKey, entry.getCacheString());
        }
        catch (Exception e) {
            this.logger.error(2201, "Error occurred while writing the cache", e);
        }
    }

    private ClientCacheState determineCacheState(Entry cachedEntry) {
        if (cachedEntry.isEmpty()) {
            return ClientCacheState.NO_FLAG_DATA;
        }
        if (this.pollingMode instanceof ManualPollingMode) {
            return ClientCacheState.HAS_CACHED_FLAG_DATA_ONLY;
        }
        if (this.pollingMode instanceof LazyLoadingMode ? cachedEntry.isExpired(System.currentTimeMillis() - (long)((LazyLoadingMode)this.pollingMode).getCacheRefreshIntervalInSeconds() * 1000L) : this.pollingMode instanceof AutoPollingMode && cachedEntry.isExpired(System.currentTimeMillis() - (long)((AutoPollingMode)this.pollingMode).getAutoPollRateInSeconds() * 1000L)) {
            return ClientCacheState.HAS_CACHED_FLAG_DATA_ONLY;
        }
        return ClientCacheState.HAS_UP_TO_DATE_FLAG_DATA;
    }
}

