/*
 * Copyright 2006-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.consol.citrus.docker.client;

import com.consol.citrus.context.TestContext;
import com.consol.citrus.docker.command.DockerCommand;
import com.consol.citrus.endpoint.AbstractEndpoint;
import com.consol.citrus.exceptions.ActionTimeoutException;
import com.consol.citrus.message.DefaultMessage;
import com.consol.citrus.message.Message;
import com.consol.citrus.message.correlation.CorrelationManager;
import com.consol.citrus.message.correlation.PollingCorrelationManager;
import com.consol.citrus.messaging.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Docker client uses Java docker client implementation for executing docker commands.
 *
 * @author Christoph Deppisch
 * @since 2.4
 */
public class DockerClient extends AbstractEndpoint implements Producer, ReplyConsumer {

    /** Logger */
    private static Logger log = LoggerFactory.getLogger(DockerClient.class);

    /** Store of reply messages */
    private CorrelationManager<DockerCommand> correlationManager;

    /**
     * Default constructor initializing endpoint configuration.
     */
    public DockerClient() {
        this(new DockerEndpointConfiguration());
    }

    /**
     * Default constructor using endpoint configuration.
     * @param endpointConfiguration
     */
    public DockerClient(DockerEndpointConfiguration endpointConfiguration) {
        super(endpointConfiguration);

        this.correlationManager = new PollingCorrelationManager<>(endpointConfiguration, "Reply message did not arrive yet");
    }

    @Override
    public DockerEndpointConfiguration getEndpointConfiguration() {
        return (DockerEndpointConfiguration) super.getEndpointConfiguration();
    }

    @Override
    public void send(Message message, TestContext context) {
        String correlationKeyName = getEndpointConfiguration().getCorrelator().getCorrelationKeyName(getName());
        String correlationKey = getEndpointConfiguration().getCorrelator().getCorrelationKey(message);
        correlationManager.saveCorrelationKey(correlationKeyName, correlationKey, context);

        if (log.isDebugEnabled()) {
            log.debug("Sending Docker request to: '" + getEndpointConfiguration().getDockerClientConfig().getDockerHost() + "'");
        }

        DockerCommand command = message.getPayload(DockerCommand.class);
        command.execute(this, context);

        log.info("Docker request was sent to endpoint: '" + getEndpointConfiguration().getDockerClientConfig().getDockerHost() + "'");

        correlationManager.store(correlationKey, command);

        if (command.getResultCallback() != null) {
            command.getResultCallback().doWithCommandResult(command.getCommandResult(), context);
        }
    }

    @Override
    public Message receive(TestContext context) {
        return receive(correlationManager.getCorrelationKey(
                getEndpointConfiguration().getCorrelator().getCorrelationKeyName(getName()), context), context);
    }

    @Override
    public Message receive(String selector, TestContext context) {
        return receive(selector, context, getEndpointConfiguration().getTimeout());
    }

    @Override
    public Message receive(TestContext context, long timeout) {
        return receive(correlationManager.getCorrelationKey(
                getEndpointConfiguration().getCorrelator().getCorrelationKeyName(getName()), context), context, timeout);
    }

    @Override
    public Message receive(String selector, TestContext context, long timeout) {
        DockerCommand command = correlationManager.find(selector, timeout);

        if (command == null) {
            throw new ActionTimeoutException("Action timeout while receiving synchronous reply message from http server");
        }

        return new DefaultMessage(command.getCommandResult());
    }

    @Override
    public Producer createProducer() {
        return this;
    }

    @Override
    public SelectiveConsumer createConsumer() {
        return this;
    }
}
