/*
 * Copyright 2006-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.consol.citrus.kafka.message;

import com.consol.citrus.context.TestContext;
import com.consol.citrus.exceptions.CitrusRuntimeException;
import com.consol.citrus.kafka.endpoint.KafkaEndpointConfiguration;
import com.consol.citrus.message.Message;
import com.consol.citrus.message.MessageConverter;
import com.consol.citrus.util.FileUtils;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.springframework.core.io.Resource;

import java.io.IOException;
import java.util.Optional;

/**
 * Basic message converter for converting Spring Integration message implementations to Kafka
 * messages and vice versa. Converter combines message converting logic and header mapping. 
 * Usually the message's payload is extracted to the Kafka message payload and default Kafka headers are mapped.
 * 
 * @author Christoph Deppisch
 * @since 2.8
 */
public class KafkaMessageConverter implements MessageConverter<ConsumerRecord<Object, Object>, ProducerRecord<Object, Object>, KafkaEndpointConfiguration> {

    @Override
    public ProducerRecord<Object, Object> convertOutbound(Message internalMessage, KafkaEndpointConfiguration endpointConfiguration, TestContext context) {
        Object payload;
        if (internalMessage.getPayload() instanceof String) {
            payload = context.replaceDynamicContentInString(internalMessage.getPayload(String.class));
        } else if (internalMessage.getPayload() instanceof Resource) {
            try {
                payload = context.replaceDynamicContentInString(FileUtils.readToString(internalMessage.getPayload(Resource.class)));
            } catch (IOException e) {
                throw new CitrusRuntimeException("Failed to read payload resource");
            }
        } else {
            payload = internalMessage.getPayload();
        }

        KafkaMessage kafkaMessage;

        if (internalMessage instanceof KafkaMessage) {
            kafkaMessage = (KafkaMessage) internalMessage;
        } else {
            kafkaMessage = new KafkaMessage(internalMessage.getPayload(), internalMessage.getHeaders());
        }

        return new ProducerRecord<>(Optional.ofNullable(kafkaMessage.getTopic()).map(context::replaceDynamicContentInString).orElse(context.replaceDynamicContentInString(endpointConfiguration.getTopic())),
                                    Optional.ofNullable(kafkaMessage.getPartition()).orElse(endpointConfiguration.getPartition()),
                                    kafkaMessage.getMessageKey(),
                                    payload,
                                    endpointConfiguration.getHeaderMapper().toHeaders(kafkaMessage.getHeaders()));
    }

    @Override
    public void convertOutbound(ProducerRecord<Object, Object> externalMessage, Message internalMessage, KafkaEndpointConfiguration endpointConfiguration, TestContext context) {
    }

    @Override
    public Message convertInbound(ConsumerRecord<Object, Object> consumerRecord, KafkaEndpointConfiguration endpointConfiguration, TestContext context) {
        if (consumerRecord == null) {
            return null;
        }
        return new KafkaMessage(consumerRecord.value(), endpointConfiguration.getHeaderMapper().fromHeaders(consumerRecord.headers()))
                                    .topic(consumerRecord.topic())
                                    .timestamp(consumerRecord.timestamp())
                                    .partition(consumerRecord.partition())
                                    .offset(consumerRecord.offset())
                                    .messageKey(consumerRecord.key());
    }
}
