/*
 * Copyright 2006-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.consol.citrus.kafka.endpoint;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import com.consol.citrus.context.TestContext;
import com.consol.citrus.exceptions.CitrusRuntimeException;
import com.consol.citrus.exceptions.MessageTimeoutException;
import com.consol.citrus.kafka.message.KafkaMessageHeaders;
import com.consol.citrus.message.Message;
import com.consol.citrus.messaging.AbstractMessageConsumer;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

/**
 * @author Christoph Deppisch
 * @since 2.8
 */
public class KafkaConsumer extends AbstractMessageConsumer {

    /** Logger */
    private static Logger log = LoggerFactory.getLogger(KafkaConsumer.class);

    /** Endpoint configuration */
    protected final KafkaEndpointConfiguration endpointConfiguration;

    /** Kafka consumer */
    private org.apache.kafka.clients.consumer.KafkaConsumer<Object, Object> consumer;

    /**
     * Default constructor using endpoint.
     * @param name
     * @param endpointConfiguration
     */
    public KafkaConsumer(String name, KafkaEndpointConfiguration endpointConfiguration) {
        super(name, endpointConfiguration);
        this.endpointConfiguration = endpointConfiguration;
        this.consumer = createConsumer();
    }

    @Override
    public Message receive(TestContext context, long timeout) {
        String topic = context.replaceDynamicContentInString(Optional.ofNullable(endpointConfiguration.getTopic())
                                                                     .orElseThrow(() -> new CitrusRuntimeException("Missing Kafka topic to receive messages from - add topic to endpoint configuration")));

        if (log.isDebugEnabled()) {
            log.debug("Receiving Kafka message on topic: '" + topic);
        }

        if (CollectionUtils.isEmpty(consumer.subscription())) {
            consumer.subscribe(Arrays.asList(StringUtils.commaDelimitedListToStringArray(topic)));
        }

        ConsumerRecords<Object, Object> records = consumer.poll(Duration.ofMillis(timeout));

        if (records.isEmpty()) {
            throw new MessageTimeoutException(timeout, topic);
        }

        if (log.isDebugEnabled()) {
            records.forEach(record -> log.debug("Received message: (" + record.key() + ", " + record.value() + ") at offset " + record.offset()));
        }

        Message received = endpointConfiguration.getMessageConverter()
                                                .convertInbound(records.iterator().next(), endpointConfiguration, context);
        context.onInboundMessage(received);

        consumer.commitSync(Duration.ofMillis(endpointConfiguration.getTimeout()));

        log.info("Received Kafka message on topic: '" + topic);
        return received;
    }

    /**
     * Stop message listener container.
     */
    public void stop() {
        try {
            if (!CollectionUtils.isEmpty(consumer.subscription())) {
                consumer.unsubscribe();
            }
        } finally {
            consumer.close(Duration.ofMillis(10 * 1000L));
        }
    }

    /**
     * Create new Kafka consumer with given endpoint configuration.
     * @return
     */
    private org.apache.kafka.clients.consumer.KafkaConsumer<Object, Object> createConsumer() {
        Map<String, Object> consumerProps = new HashMap<>();
        consumerProps.put(ConsumerConfig.CLIENT_ID_CONFIG, Optional.ofNullable(endpointConfiguration.getClientId()).orElse(KafkaMessageHeaders.KAFKA_PREFIX + "consumer_" + UUID.randomUUID().toString()));
        consumerProps.put(ConsumerConfig.GROUP_ID_CONFIG, endpointConfiguration.getConsumerGroup());
        consumerProps.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, Optional.ofNullable(endpointConfiguration.getServer()).orElse("localhost:9092"));
        consumerProps.put(ConsumerConfig.MAX_POLL_RECORDS_CONFIG, 1);
        consumerProps.put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, endpointConfiguration.isAutoCommit());
        consumerProps.put(ConsumerConfig.AUTO_COMMIT_INTERVAL_MS_CONFIG, endpointConfiguration.getAutoCommitInterval());
        consumerProps.put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, endpointConfiguration.getOffsetReset());
        consumerProps.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, endpointConfiguration.getKeyDeserializer());
        consumerProps.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG, endpointConfiguration.getValueDeserializer());

        consumerProps.putAll(endpointConfiguration.getConsumerProperties());

        return new org.apache.kafka.clients.consumer.KafkaConsumer<>(consumerProps);
    }

    /**
     * Sets the consumer.
     *
     * @param consumer
     */
    public void setConsumer(org.apache.kafka.clients.consumer.KafkaConsumer<Object, Object> consumer) {
        this.consumer = consumer;
    }
}
