/*
 * Copyright 2021 the original author or authors.
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.consol.citrus.dsl;

import java.util.LinkedHashMap;
import java.util.Map;

import com.consol.citrus.builder.PathExpressionAdapter;
import com.consol.citrus.builder.WithExpressions;
import com.consol.citrus.message.MessageProcessor;
import com.consol.citrus.validation.context.ValidationContext;
import com.consol.citrus.validation.json.JsonPathMessageProcessor;
import com.consol.citrus.validation.json.JsonPathMessageValidationContext;
import com.consol.citrus.validation.json.JsonPathVariableExtractor;
import com.consol.citrus.variable.VariableExtractor;

/**
 * @author Christoph Deppisch
 */
public class JsonPathSupport implements WithExpressions<JsonPathSupport>, PathExpressionAdapter {

    private final Map<String, Object> expressions = new LinkedHashMap<>();

    /**
     * Static entrance for all JsonPath related Java DSL functionalities.
     * @return
     */
    public static JsonPathSupport jsonPath() {
        return new JsonPathSupport();
    }

    @Override
    public MessageProcessor asProcessor() {
        return new JsonPathMessageProcessor.Builder()
                .expressions(expressions)
                .build();
    }

    @Override
    public VariableExtractor asExtractor() {
        return new JsonPathVariableExtractor.Builder()
                .expressions(expressions)
                .build();
    }

    @Override
    public ValidationContext asValidationContext() {
        return new JsonPathMessageValidationContext.Builder()
                .expressions(expressions)
                .build();
    }

    @Override
    public JsonPathSupport expressions(Map<String, Object> expressions) {
        this.expressions.putAll(expressions);
        return this;
    }

    @Override
    public JsonPathSupport expression(String expression, Object value) {
        expressions.put(expression, value);
        return this;
    }
}
