/*
 * Copyright 2006-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.consol.citrus.validation.json.report;

import com.github.fge.jsonschema.core.exceptions.ProcessingException;
import com.github.fge.jsonschema.core.report.LogLevel;
import com.github.fge.jsonschema.core.report.ProcessingMessage;
import com.github.fge.jsonschema.core.report.ProcessingReport;
import com.google.common.collect.Iterators;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * This class implements a report that represents a gracious interpretation of a set of processing reports
 * generated by the json-schema-validation library.
 * @since 2.7.3
 */
public class GraciousProcessingReport implements ProcessingReport {

    /**
     * The highest log level seen so far
     */
    private LogLevel currentLevel = LogLevel.DEBUG;

    /**
     * The log threshold
     */
    private LogLevel logLevel = LogLevel.INFO;

    /**
     * The exception threshold
     */
    private LogLevel exceptionThreshold = LogLevel.FATAL;

    private boolean success;

    private final List<ProcessingMessage> messages = new ArrayList<>();

    /**
     * Constructor to determine the default success state oft he Report
     * @param success the default success state
     */
    public GraciousProcessingReport(boolean success) {
        this.success = success;
    }

    /**
     * Creates a GraciousProcessingReport while preserving the information from the given list of ProcessingReports
     * @param processingReports The list of reports to merge with the new GraciousProcessingReport
     */
    public GraciousProcessingReport(List<ProcessingReport> processingReports) {
        this(false);
        for (ProcessingReport processingReport : processingReports) {
            mergeWith(processingReport);
        }
    }

    @Override
    public void mergeWith(final ProcessingReport processingReport) {
        success = success || processingReport.isSuccess();
        processingReport.forEach(messages::add);
    }

    @Override
    public LogLevel getLogLevel() {
        return logLevel;
    }

    @Override
    public LogLevel getExceptionThreshold() {
        return exceptionThreshold;
    }

    @Override
    public void debug(ProcessingMessage message) throws ProcessingException {
        dispatch(message.setLogLevel(LogLevel.DEBUG));
    }

    @Override
    public void info(ProcessingMessage message) throws ProcessingException {
        dispatch(message.setLogLevel(LogLevel.INFO));
    }

    @Override
    public void warn(ProcessingMessage message) throws ProcessingException {
        dispatch(message.setLogLevel(LogLevel.WARNING));
    }

    @Override
    public void error(ProcessingMessage message) throws ProcessingException {
        dispatch(message.setLogLevel(LogLevel.ERROR));
    }

    @Override
    public void fatal(ProcessingMessage message) throws ProcessingException {
        dispatch(message.setLogLevel(LogLevel.FATAL));
    }

    public boolean isSuccess() {
        return success;
    }

    public List<ProcessingMessage> getMessages() {
        return messages;
    }

    @Override
    @Nonnull
    public Iterator<ProcessingMessage> iterator() {
        return Iterators.unmodifiableIterator(messages.iterator());
    }

    /**
     * Main dispatch method
     *
     * All messages logged go through this method. According to the report
     * configuration, the message will either be ignored, logged or raise an
     * exception.
     *
     * @param message the message to log
     * @throws ProcessingException the message's level and report configuration
     * require that an exception be thrown
     */
    private void dispatch(final ProcessingMessage message) throws ProcessingException {
        final LogLevel level = message.getLogLevel();

        if (level.compareTo(exceptionThreshold) >= 0)
            throw message.asException();
        if (level.compareTo(currentLevel) > 0)
            currentLevel = level;
        if (level.compareTo(logLevel) >= 0)
            log(message);
    }

    public void log(final ProcessingMessage message) {
        messages.add(message);
    }
}
