/*
 * Decompiled with CFR 0.152.
 */
package com.conveyal.gtfs;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.conveyal.gtfs.GTFSFeed;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.io.ByteStreams;
import com.google.common.io.Files;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.NoSuchElementException;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.zip.ZipFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GTFSCache {
    private static final Logger LOG = LoggerFactory.getLogger(GTFSCache.class);
    public final String bucket;
    public final String bucketFolder;
    public final File cacheDir;
    private static final AmazonS3 s3 = new AmazonS3Client();
    RemovalListener<String, GTFSFeed> removalListener = new RemovalListener<String, GTFSFeed>(){

        public void onRemoval(RemovalNotification<String, GTFSFeed> removalNotification) {
            if (GTFSCache.this.bucket != null) {
                String[] extensions;
                String id = (String)removalNotification.getKey();
                for (String type : extensions = new String[]{".db", ".db.p", ".zip"}) {
                    File file = new File(GTFSCache.this.cacheDir, id + type);
                    file.delete();
                }
            }
        }
    };
    private LoadingCache<String, GTFSFeed> cache = CacheBuilder.newBuilder().maximumSize(10L).removalListener(this.removalListener).build((CacheLoader)new CacheLoader<String, GTFSFeed>(){

        public GTFSFeed load(String s) throws Exception {
            return GTFSCache.this.retrieveFeed(s);
        }
    });

    public GTFSCache(String bucket, File cacheDir) {
        if (bucket == null) {
            LOG.info("No bucket specified; GTFS Cache will run locally");
        } else {
            LOG.info("Using bucket {} for GTFS Cache", (Object)bucket);
        }
        this.bucket = bucket;
        this.bucketFolder = null;
        this.cacheDir = cacheDir;
    }

    public GTFSCache(String bucket, String bucketFolder, File cacheDir) {
        if (bucket == null) {
            LOG.info("No bucket specified; GTFS Cache will run locally");
        } else {
            LOG.info("Using bucket {} for GTFS Cache", (Object)bucket);
        }
        this.bucket = bucket;
        this.bucketFolder = bucketFolder != null ? bucketFolder.replaceAll("\\/", "") : null;
        this.cacheDir = cacheDir;
    }

    public GTFSFeed put(String id, File feedFile) throws Exception {
        return this.put(id, feedFile, null);
    }

    public GTFSFeed put(Function<GTFSFeed, String> idGenerator, File feedFile) throws Exception {
        return this.put(null, feedFile, idGenerator);
    }

    private GTFSFeed put(String id, File feedFile, Function<GTFSFeed, String> idGenerator) throws Exception {
        String tempId = id != null ? id : UUID.randomUUID().toString();
        String cleanTempId = GTFSCache.cleanId(tempId);
        File dbFile = new File(this.cacheDir, cleanTempId + ".db");
        File movedFeedFile = new File(this.cacheDir, cleanTempId + ".zip");
        if (!feedFile.equals(movedFeedFile)) {
            Files.copy((File)feedFile, (File)movedFeedFile);
        }
        GTFSFeed feed = new GTFSFeed(dbFile.getAbsolutePath());
        feed.loadFromFile(new ZipFile(movedFeedFile));
        feed.findPatterns();
        if (idGenerator != null) {
            id = idGenerator.apply(feed);
        }
        String cleanId = GTFSCache.cleanId(id);
        feed.close();
        if (idGenerator != null) {
            new File(this.cacheDir, cleanTempId + ".zip").renameTo(new File(this.cacheDir, cleanId + ".zip"));
            new File(this.cacheDir, cleanTempId + ".db").renameTo(new File(this.cacheDir, cleanId + ".db"));
            new File(this.cacheDir, cleanTempId + ".db.p").renameTo(new File(this.cacheDir, cleanId + ".db.p"));
        }
        if (this.bucket != null) {
            String key;
            LOG.info("Writing feed to s3 cache");
            String string = key = this.bucketFolder != null ? String.join((CharSequence)"/", this.bucketFolder, cleanId) : cleanId;
            if (!s3.doesObjectExist(this.bucket, key + ".zip")) {
                s3.putObject(this.bucket, key + ".zip", feedFile);
                LOG.info("Zip file written.");
            } else {
                LOG.info("Zip file already exists on s3.");
            }
            s3.putObject(this.bucket, key + ".db", new File(this.cacheDir, cleanId + ".db"));
            s3.putObject(this.bucket, key + ".db.p", new File(this.cacheDir, cleanId + ".db.p"));
            LOG.info("db files written.");
        }
        feed = new GTFSFeed(new File(this.cacheDir, cleanId + ".db").getAbsolutePath());
        this.cache.put((Object)id, (Object)feed);
        return feed;
    }

    public GTFSFeed get(String id) {
        try {
            return (GTFSFeed)this.cache.get((Object)id);
        }
        catch (ExecutionException e) {
            LOG.error("Error loading local MapDB.", (Throwable)e);
            this.deleteLocalDBFiles(id);
            return null;
        }
    }

    public boolean containsId(String id) {
        GTFSFeed feed;
        try {
            feed = (GTFSFeed)this.cache.get((Object)id);
        }
        catch (Exception e) {
            return false;
        }
        return feed != null;
    }

    private GTFSFeed retrieveFeed(String originalId) {
        GTFSFeed feed;
        String id = GTFSCache.cleanId(originalId);
        String key = this.bucketFolder != null ? String.join((CharSequence)"/", this.bucketFolder, id) : id;
        File dbFile = new File(this.cacheDir, id + ".db");
        if (dbFile.exists()) {
            LOG.info("Processed GTFS was found cached locally");
            try {
                feed = new GTFSFeed(dbFile.getAbsolutePath());
                if (feed != null) {
                    return feed;
                }
            }
            catch (Exception e) {
                LOG.warn("Error loading local MapDB.", (Throwable)e);
                this.deleteLocalDBFiles(id);
            }
        }
        if (this.bucket != null) {
            try {
                LOG.info("Attempting to download cached GTFS MapDB.");
                S3Object db = s3.getObject(this.bucket, key + ".db");
                S3ObjectInputStream is = db.getObjectContent();
                FileOutputStream fos = new FileOutputStream(dbFile);
                ByteStreams.copy((InputStream)is, (OutputStream)fos);
                is.close();
                fos.close();
                S3Object dbp = s3.getObject(this.bucket, key + ".db.p");
                S3ObjectInputStream isp = dbp.getObjectContent();
                FileOutputStream fosp = new FileOutputStream(new File(this.cacheDir, id + ".db.p"));
                ByteStreams.copy((InputStream)isp, (OutputStream)fosp);
                isp.close();
                fosp.close();
                LOG.info("Returning processed GTFS from S3");
                feed = new GTFSFeed(dbFile.getAbsolutePath());
                if (feed != null) {
                    return feed;
                }
            }
            catch (AmazonS3Exception e) {
                LOG.warn("DB file for key {} does not exist on S3.", (Object)key);
            }
            catch (IOException | ExecutionException e) {
                LOG.warn("Error retrieving MapDB from S3, will load from original GTFS.", (Throwable)e);
            }
        }
        LOG.info("Loading feed from local cache directory...");
        File feedFile = new File(this.cacheDir, id + ".zip");
        if (!feedFile.exists() && this.bucket != null) {
            LOG.info("Feed not found locally, downloading from S3.");
            try {
                S3Object gtfs = s3.getObject(this.bucket, key + ".zip");
                S3ObjectInputStream is = gtfs.getObjectContent();
                FileOutputStream fos = new FileOutputStream(feedFile);
                ByteStreams.copy((InputStream)is, (OutputStream)fos);
                is.close();
                fos.close();
            }
            catch (Exception e) {
                LOG.error("Could not download feed at s3://{}/{}.", (Object)this.bucket, (Object)key);
                throw new RuntimeException(e);
            }
        }
        if (feedFile.exists()) {
            try {
                return this.put(originalId, feedFile);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        throw new NoSuchElementException(originalId);
    }

    private void deleteLocalDBFiles(String id) {
        String[] extensions;
        for (String type : extensions = new String[]{".db", ".db.p"}) {
            File file = new File(this.cacheDir, id + type);
            file.delete();
        }
    }

    public static String cleanId(String id) {
        return id.replaceAll("[^A-Za-z0-9_]", "-");
    }
}

