/*
 **************************************************************************
 * 版权声明：
 * 本软件为木偶人(http://www.cottsoft.com)开发研制。未经本人正式书面同意，
 * 其他任何个人、团体不得将本软件使用、复制、修改或发布于商业产品中。 
 **************************************************************************
 * 程序描述:
 * XML加载类及处理工具类。
 * 
 **************************************************************************
 * 修改历史:  
 * Date:       	    by:    		    Reason:  
 *           
 * 2016年11月15日    Simon.Hoo		Initial Version.
 *************************************************************************
 */
package com.cottsoft.framework.xml;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.springframework.util.StringUtils;

import com.cottsoft.framework.constant.code.EECode;
import com.cottsoft.framework.exception.FrameworkException;
import com.cottsoft.framework.util.FileUtil;


/**
 * @author Simon.Hoo
 * @version v1.0.0
 * @since JDK1.7
 * XML加载类及处理工具类。
 */
public class XmlLoader {
	private static Map<String, Document> xmlMap = new HashMap<String, Document>(16);
	
	private Map<String, String> xmlFiles;
	
	private String xmlRoot;
	
	public XmlLoader(){}
	
	/**
	 * Description：<br> 
	 * 文件列表参数的构造方法
	 * @param xmlFiles
	 */
	public XmlLoader(Map<String, String> xmlFiles) {
		this.xmlFiles = xmlFiles;
		loadXmlWithMap(xmlFiles);
	}
	
	/**
	 * Description：<br> 
	 * 文件根目录构造方法
	 * @param xmlRoot
	 */
	public XmlLoader(String xmlRoot) {
		this.xmlRoot = xmlRoot;
		loadXmlWithRoot(xmlRoot);
	}
	
	/**
	 * Description：<br> 
	 * 使用Dom4j加载XML到内存中，加载的文件为传入MAP中的文件,缓存中KEY的值为传入MAP的KEY值
	 * @param xmlFiles
	 */
	@SuppressWarnings("unchecked")
	public void loadXmlWithMap(Map<String, String> xmlFiles) {
		Set<?> processorSet = xmlFiles.entrySet();
		Iterator<?> it = processorSet.iterator();
		SAXReader reader = new SAXReader();
		
		//将XML注入到MAP中
		while (it.hasNext()) {
			Map.Entry<String, ?> entry = (Entry<String, ?>) it.next();
			try {
				xmlMap.put(entry.getKey(),reader.read((String) entry.getValue()));
			} catch (DocumentException e) {
				e.printStackTrace();
			}
		}
	}
	
	/**
	 * Description：<br> 
	 * 使用Dom4j加载XML到内存中,加载的文件为传入目录中的所有XML文件，缓存中KEY的值为XML根节点NAME属性的值
	 * @param xmlRoot
	 */
	public void loadXmlWithRoot(String xmlRoot) {
		SAXReader reader = null;
		try{
			if(!FileUtil.isExist(xmlRoot)){
				throw new FrameworkException(EECode.E00001);
			}
			
			File dir = new File(xmlRoot);
			
			if(!dir.isDirectory()){
				throw new FrameworkException(EECode.E00002);
			}
			String[] tempList = dir.list();

			File temp = null;
			
			for (int i = 0; i < tempList.length; i++) {
				if (xmlRoot.endsWith(File.separator)) {
					temp = new File(xmlRoot + tempList[i]);
				} else {
					temp = new File(xmlRoot + File.separator + tempList[i]);
				}
				
				if (temp.isFile() && temp.getName().toLowerCase().endsWith(".xml")) {
					reader = new SAXReader();
					Document xmlDoc = reader.read(temp);
					
					String name = getAttributeValue(xmlDoc.getRootElement(), "NAME");
					
					if(!StringUtils.isEmpty(name)){
						xmlMap.put(name.trim(),xmlDoc);
					}
				}else if (temp.isDirectory()) {
					loadXmlWithRoot(xmlRoot+ "/" + tempList[i]);
				}
			}

		}catch(Exception e){
			e.printStackTrace();
		}
	}
	
	/**
	 * Description：<br> 
	 * 从一个Document对象的节点Element中找到指定的节点对象
	 * @param searchedElement 被找的节点对象，源对象
	 * @param targetNodePrefix 要找的节点名称
	 * @param targetNodeAttributeName 要找的节点属性名称
	 * @param targetNodeAttributeValue 要找的节点履性值
	 * @return element
	 */
	public Element findElement(Element searchedElement,
			String targetNodePrefix, String targetNodeAttributeName,
			String targetNodeAttributeValue) {
		Element elementTarget = null;
		for (Iterator<?> i = searchedElement.elementIterator(targetNodePrefix); i.hasNext();) {
			Element element = (Element) i.next();
			String tmpAttrValue = getAttributeValue(element,targetNodeAttributeName);
			if (tmpAttrValue.equals(targetNodeAttributeValue)) {
				elementTarget = element;
				break;
			}
		}
		return elementTarget;
	}
	
	/**
	 * Description：<br> 
	 * 从一个Document对象的节点Element中找到指定的节点对象
	 * @param searchedElement 被找的节点对象，源对象
	 * @param targetNodePrefix 要找的节点名称
	 * @param targetNodeAttributeName 要找的节点属性名称
	 * @param targetNodeAttributeValue 要找的节点履性值
	 * @return list
	 */
	public List<Element> findElementList(Element searchedElement,
			String targetNodePrefix, String targetNodeAttributeName,
			String targetNodeAttributeValue) {
		List<Element> elementTargetList = new ArrayList<Element>();
		for (Iterator<?> i = searchedElement.elementIterator(targetNodePrefix); i.hasNext();) {
			Element element = (Element) i.next();
			String tmpAttrValue = getAttributeValue(element,targetNodeAttributeName);
			if (tmpAttrValue.equals(targetNodeAttributeValue)) {
				elementTargetList.add(element);
			}
		}
		return elementTargetList;
	}
	
	/**
	 * Description：<br> 
	 * 用于从一个节点中定向找其子节点，通常同名的子节不会出现多个，如果出现多个，则返回第一个
	 * @param searchedElement
	 * @param targetNodePrefix
	 * @return element
	 */
	public Element findElement(Element searchedElement,String targetNodePrefix) {
		Element elementTarget = null;
		
		for (Iterator<?> i = searchedElement.elementIterator(targetNodePrefix); i.hasNext();) {
			elementTarget = (Element) i.next();
			break;
		}
		return elementTarget;
	}
	
	/**
	 * Description：<br> 
	 * 根据节点属性名获取属性值
	 * @param element
	 * @param attributeName
	 * @return string
	 */
	public String getAttributeValue(Element element, String attributeName) {
		String attributeValue = null;
		for (Iterator<?> i = element.attributeIterator(); i.hasNext();) {
			Attribute attribute = (Attribute) i.next();
			if (attribute.getName().equals(attributeName)) {
				attributeValue = (String) attribute.getData();
				break;
			}
		}
		return attributeValue;
	}
	
	/**
	 * Description：<br> 
	 * 刷新已经加载的XML模板。
	 */
	public void flush() {
		//未实现
	}
	
	/**
	 * Description：<br> 
	 * 从MAP中删除一个已存在的Document对象
	 * @param key
	 */
	public void removeXml(String key) {
		if (xmlMap.get(key) != null) {
			xmlMap.remove(key);
		}
	}
	
	/**
	 * Description：<br> 
	 * 清空缓存中的Document对象
	 */
	public void removeAll() {
		xmlMap.clear();
	}
	
	/**
	 * Description：<br> 
	 * 根据Key值从缓存中获取Document对象
	 * @param key
	 * @return document
	 */
	public Document getXml(String key) {
		return xmlMap.get(key);
	}
	
	public static void setXmlMap(Map<String, Document> xmlMap) {
		XmlLoader.xmlMap = xmlMap;
	}

	public Map<String, String> getXmlFiles() {
		return xmlFiles;
	}

	public void setXmlFiles(Map<String, String> xmlFiles) {
		this.xmlFiles = xmlFiles;
	}

	public String getXmlRoot() {
		return xmlRoot;
	}

	public void setXmlRoot(String xmlRoot) {
		this.xmlRoot = xmlRoot;
	}
}
