/*
 * Copyright (c) 2016 Couchbase, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.couchbase.client.java.query;

import java.util.Iterator;
import java.util.List;

import com.couchbase.client.core.annotations.InterfaceAudience;
import com.couchbase.client.core.annotations.InterfaceStability;
import com.couchbase.client.java.document.json.JsonArray;
import com.couchbase.client.java.document.json.JsonObject;

/**
 * Represents the results of a {@link N1qlQuery}, in a blocking fashion.
 * Note that the result is complete, meaning it will block until all
 * data has been streamed from the server.
 *
 * @author Michael Nitschinger
 * @since 2.0.0
 */
@InterfaceStability.Experimental
@InterfaceAudience.Public
public interface N1qlQueryResult extends Iterable<N1qlQueryRow> {

    /**
     * @return the list of all {@link N1qlQueryRow}, the results of the query, if successful.
     */
    List<N1qlQueryRow> allRows();

    /**
     * @return an iterator over the list of all {@link N1qlQueryRow}, the results of the query, if successful.
     */
    Iterator<N1qlQueryRow> rows();

    /**
     * @return an object representing the signature of the results, that can be used to
     * learn about the common structure of each {@link #rows() row}. This signature is usually a
     * {@link JsonObject}, but could also be any JSON-valid type like a boolean scalar, {@link JsonArray}...
     */
    Object signature();

    /**
     * @return an object describing some metrics/info about the execution of the query.
     */
    N1qlMetrics info();

    /**
     * @return true if the query could be parsed, false if it short-circuited due to syntax/fatal error.
     */
    boolean parseSuccess();

    /**
     * Denotes the success or failure of the query. It could fail slower than with
     * {@link #parseSuccess()}, for example if a fatal error comes up while streaming the results
     * to the client. This method blocks until the query is over and the success can be established.
     */
    boolean finalSuccess();

    /**
     * Returns the final status of the query. For example, a successful query will return "<code>success</code>"
     * (which is equivalent to {@link #finalSuccess()} returning true). Other statuses include (but are not limited to)
     * "<code>fatal</code>" when fatal errors occurred and "<code>timeout</code>" when the query timed out on the server
     * side but not yet on the client side. This method blocks until the query is over and the status can be established.
     */
    String status();

    /**
     * @return A list of errors or warnings encountered while executing the query.
     */
    List<JsonObject> errors();

    /**
     * @return the requestId generated by the server
     */
    String requestId();

    /**
     * @return the clientContextId that was set by the client (could be truncated to 64 bytes of UTF-8 chars)
     */
    String clientContextId();

}