/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.connect.kafka.util;

import com.couchbase.client.core.annotation.Stability;
import com.couchbase.connect.kafka.config.sink.CouchbaseSinkConfig;
import com.couchbase.connect.kafka.config.sink.SinkBehaviorConfig;
import com.couchbase.connect.kafka.util.config.ConfigHelper;
import java.io.Closeable;
import java.time.Duration;
import java.util.Objects;
import org.apache.kafka.connect.errors.RetriableException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Stability.Internal
public class KafkaRetryHelper
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(KafkaRetryHelper.class);
    private final ThreadLocal<Deadline> deadline = new ThreadLocal();
    private final Duration retryTimeout;
    private final Clock clock;
    private final String actionDescription;

    public KafkaRetryHelper(String actionDescription, Duration retryTimeout) {
        this(actionDescription, retryTimeout, System::nanoTime);
    }

    KafkaRetryHelper(String actionDescription, Duration retryTimeout, Clock clock) {
        this.actionDescription = Objects.requireNonNull(actionDescription);
        this.retryTimeout = Duration.ofNanos(KafkaRetryHelper.toNanosSaturated(retryTimeout));
        this.clock = Objects.requireNonNull(clock);
        log.info("Initialized retry helper for {} with timeout {}", (Object)actionDescription, (Object)this.retryTimeout);
    }

    private static long toNanosSaturated(Duration d) {
        try {
            return d.toNanos();
        }
        catch (ArithmeticException e) {
            return Long.MAX_VALUE;
        }
    }

    public void runWithRetry(Runnable r) {
        try {
            if (this.deadline.get() != null) {
                log.info("Retrying {}", (Object)this.actionDescription);
            }
            r.run();
            if (this.deadline.get() != null) {
                this.deadline.remove();
                log.info("Retry for {} succeeded.", (Object)this.actionDescription);
            }
        }
        catch (Exception e) {
            if (this.retryTimeout.isZero()) {
                String retryTimeoutName = ConfigHelper.keyName(CouchbaseSinkConfig.class, SinkBehaviorConfig::retryTimeout);
                log.error("Initial attempt for {} failed. Retry is disabled. Connector will terminate. To mitigate this kind of failure, enable retry by setting the '{}' connector config property.", new Object[]{this.actionDescription, retryTimeoutName, e});
                throw e;
            }
            if (this.deadline.get() == null) {
                this.deadline.set(new Deadline(this.clock, this.retryTimeout));
                throw new RetriableException("Initial attempt for " + this.actionDescription + " failed. Will try again later.", (Throwable)e);
            }
            if (this.deadline.get().hasTimeLeft()) {
                throw new RetriableException("Retry for " + this.actionDescription + " failed. Will try again later.", (Throwable)e);
            }
            log.error("Retry for {} failed. Retry timeout ({}) expired. Connector will terminate.", new Object[]{this.actionDescription, this.retryTimeout, e});
            throw e;
        }
    }

    @Override
    public void close() {
        this.deadline.remove();
    }

    @FunctionalInterface
    static interface Clock {
        public long nanoTime();
    }

    static class Deadline {
        private final Clock clock;
        private final long startNanos;
        private final long durationNanos;

        Deadline(Clock clock, Duration d) {
            this.clock = Objects.requireNonNull(clock);
            this.startNanos = clock.nanoTime();
            this.durationNanos = d.toNanos();
        }

        public boolean hasTimeLeft() {
            return this.clock.nanoTime() - this.startNanos < this.durationNanos;
        }
    }
}

