//
// Copyright (c) 2020 Couchbase, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at:
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package com.couchbase.lite;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.File;

import com.couchbase.lite.internal.ImmutableDatabaseConfiguration;
import com.couchbase.lite.internal.core.C4Constants;
import com.couchbase.lite.internal.utils.Preconditions;


/**
 * A Couchbase Lite database.
 */
public final class Database extends AbstractDatabase {
    /**
     * Make a copy of a database in a new location.
     * It is recommended that this method not be used on an open database.
     *
     * @param path   path to the existing db file
     * @param name   the name of the new DB
     * @param config a config with the new location
     * @throws CouchbaseLiteException on copy failure
     */
    public static void copy(@NonNull File path, @NonNull String name, @NonNull DatabaseConfiguration config)
        throws CouchbaseLiteException {
        Preconditions.assertNotNull(config, "config");
        AbstractDatabase.copy(path, name, config.getDirectory(), C4Constants.EncryptionAlgorithm.NONE, null);
    }

    static void copy(@NonNull File path, @NonNull String name, @NonNull ImmutableDatabaseConfiguration config)
        throws CouchbaseLiteException {
        AbstractDatabase.copy(path, name, config.getDirectory(), C4Constants.EncryptionAlgorithm.NONE, null);
    }


    //---------------------------------------------
    // Constructors
    //---------------------------------------------

    /**
     * Construct a Database with a given name and the default config.
     * If the database does not yet exist it will be created.
     *
     * @param name   The name of the database: May NOT contain capital letters!
     * @throws CouchbaseLiteException if any error occurs during the open operation.
     */
    public Database(@NonNull String name) throws CouchbaseLiteException {super(name); }

    /**
     * Construct a  AbstractDatabase with a given name and database config.
     * If the database does not yet exist, it will be created, unless the `readOnly` option is used.
     *
     * @param name   The name of the database: May NOT contain capital letters!
     * @param config The database config.
     * @throws CouchbaseLiteException Throws an exception if any error occurs during the open operation.
     */
    public Database(@NonNull String name, @NonNull DatabaseConfiguration config) throws CouchbaseLiteException {
        super(name, config);
    }

    Database(@NonNull String name, @NonNull ImmutableDatabaseConfiguration config) throws CouchbaseLiteException {
        super(name, config);
    }

    //---------------------------------------------
    // Protected
    //---------------------------------------------

    @NonNull
    @Override
    protected Database getDatabase() { return this; }

    //---------------------------------------------
    // Package visible
    //---------------------------------------------

    // Implementation of abstract methods for Encryption
    @Override
    int getEncryptionAlgorithm() { return C4Constants.EncryptionAlgorithm.NONE; }

    @Nullable
    @Override
    byte[] getEncryptionKey() { return null; }
}
