/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.lite.internal;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.couchbase.lite.CouchbaseLiteError;
import com.couchbase.lite.LogDomain;
import com.couchbase.lite.internal.logging.Log;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.attribute.PosixFilePermission;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.atomic.AtomicBoolean;

final class NativeLibrary {
    private static final String JAVA_PATH_SEPARATOR = "/";
    private static final String RESOURCE_BASE_DIR = "libs";
    private static final String OS_WINDOWS = "windows";
    private static final String OS_LINUX = "linux";
    private static final String OS_MAC = "mac";
    private static final String ARCH_X86 = "x86_64";
    private static final String ARCH_APPLE_ARM = "aarch64";
    private static final String ARCH_UNIVERSAL = "universal";
    private static final String LIB_JNI = "LiteCoreJNI";
    private static final String LIB_LITE_CORE = "LiteCore";
    private static final String WINDOWS_OS_DIR = "windows";
    private static final String LINUX_OS_DIR = "linux";
    private static final String MAC_OS_DIR = "macos";
    private static final String LIB_DIR = "/lib";
    private static final String DIGEST_MD5 = ".MD5";
    private static final AtomicBoolean LOADED = new AtomicBoolean();

    private NativeLibrary() {
    }

    public static boolean isWindows() {
        return NativeLibrary.isWindows(System.getProperty("os.name"));
    }

    public static boolean isMacOS() {
        return NativeLibrary.isMacOS(System.getProperty("os.name"));
    }

    public static boolean isLinux() {
        return NativeLibrary.isLinux(System.getProperty("os.name"));
    }

    static void load(@NonNull File scratchDir) {
        if (LOADED.getAndSet(true)) {
            return;
        }
        String os = System.getProperty("os.name");
        String arch = System.getProperty("os.arch");
        NativeLibrary.loadLibrary(LIB_LITE_CORE, NativeLibrary.getCoreArchDir(os, arch), os, arch, scratchDir);
        NativeLibrary.loadLibrary(LIB_JNI, NativeLibrary.getJniArchDir(os, arch), os, arch, scratchDir);
    }

    @NonNull
    static String loadExtension(@NonNull String libName, @NonNull File scratchDir, @Nullable List<String> supportLibNames) throws IOException {
        String os = System.getProperty("os.name");
        String libExtension = NativeLibrary.getLibExtension(os);
        String extLib = libName + libExtension;
        String resDirPath = NativeLibrary.getCoreArchDir(os, System.getProperty("os.arch"));
        File targetDir = NativeLibrary.computeTargetDirectory(scratchDir, resDirPath, extLib);
        NativeLibrary.extract(extLib, resDirPath, targetDir);
        if (supportLibNames != null) {
            for (String supportLib : supportLibNames) {
                try {
                    NativeLibrary.extract(supportLib + libExtension, resDirPath, targetDir);
                }
                catch (IOException err) {
                    Log.d(LogDomain.DATABASE, "Failed loading extension support lib: %s", err, supportLib);
                }
            }
        }
        return targetDir.getCanonicalPath();
    }

    private static void loadLibrary(@NonNull String libName, @NonNull String resDirPath, @NonNull String os, @NonNull String arch, @NonNull File scratchDir) {
        String libPath;
        String lib = System.mapLibraryName(libName);
        File targetDir = null;
        try {
            targetDir = NativeLibrary.computeTargetDirectory(scratchDir, resDirPath, lib);
            libPath = NativeLibrary.extract(lib, resDirPath, targetDir);
            if (!NativeLibrary.isWindows(os)) {
                NativeLibrary.setPermissions(libPath);
            }
        }
        catch (IOException e) {
            throw new CouchbaseLiteError("Failed extracting library resource: " + lib + " to " + targetDir, e);
        }
        try {
            System.load(libPath);
        }
        catch (Throwable e) {
            Exception err = e instanceof Exception ? (Exception)e : new Exception(e);
            throw new CouchbaseLiteError("Failed loading native library " + lib + " @" + libPath + " (" + os + JAVA_PATH_SEPARATOR + arch + ")", err);
        }
    }

    @NonNull
    private static File computeTargetDirectory(@NonNull File scratchDir, @NonNull String resDirPath, @NonNull String lib) throws IOException {
        String path = resDirPath + JAVA_PATH_SEPARATOR + lib + DIGEST_MD5;
        try (InputStream rezStream = NativeLibrary.class.getResourceAsStream(path);){
            String hash;
            if (rezStream == null) {
                throw new IOException("Cannot find digest resource");
            }
            try (BufferedReader in = new BufferedReader(new InputStreamReader(rezStream, StandardCharsets.UTF_8));){
                hash = in.readLine();
            }
            if (hash == null) {
                throw new IOException("Digest file is empty");
            }
            File file = new File(scratchDir, hash.trim()).getCanonicalFile();
            return file;
        }
    }

    @NonNull
    private static String getOsDir(@NonNull String osName) {
        if (NativeLibrary.isWindows(osName)) {
            return "windows";
        }
        if (NativeLibrary.isLinux(osName)) {
            return "linux";
        }
        if (NativeLibrary.isMacOS(osName)) {
            return MAC_OS_DIR;
        }
        throw new CouchbaseLiteError("Unsupported OS: " + osName);
    }

    @NonNull
    private static String getCoreArchDir(@NonNull String osName, @NonNull String archName) {
        String rootPath = NativeLibrary.getRootPath(osName);
        if (NativeLibrary.isWindows(osName) || NativeLibrary.isLinux(osName)) {
            return rootPath + ARCH_X86 + LIB_DIR;
        }
        if (NativeLibrary.isMacOS(osName)) {
            return rootPath + ARCH_UNIVERSAL + LIB_DIR;
        }
        throw new CouchbaseLiteError("Unsupported LiteCore architecture: " + osName + JAVA_PATH_SEPARATOR + archName);
    }

    @NonNull
    private static String getJniArchDir(@NonNull String osName, @NonNull String archName) {
        String rootPath = NativeLibrary.getRootPath(osName);
        if (NativeLibrary.isWindows(osName) || NativeLibrary.isLinux(osName)) {
            return rootPath + ARCH_X86 + LIB_DIR;
        }
        String arch = archName.toLowerCase(Locale.getDefault());
        if (NativeLibrary.isMacOS(osName) && (ARCH_X86.equals(arch) || ARCH_APPLE_ARM.equals(arch))) {
            return rootPath + arch + LIB_DIR;
        }
        throw new CouchbaseLiteError("Unsupported JNI architecture: " + osName + JAVA_PATH_SEPARATOR + archName);
    }

    @NonNull
    private static String getLibExtension(@NonNull String osName) {
        if (NativeLibrary.isWindows(osName)) {
            return ".dll";
        }
        if (NativeLibrary.isMacOS(osName)) {
            return ".dylib";
        }
        if (NativeLibrary.isLinux(osName)) {
            return ".so";
        }
        throw new CouchbaseLiteError("Unsupported OS: " + osName);
    }

    private static void setPermissions(String targetPath) throws IOException {
        HashSet<PosixFilePermission> perms = new HashSet<PosixFilePermission>();
        perms.add(PosixFilePermission.OWNER_READ);
        perms.add(PosixFilePermission.OWNER_WRITE);
        perms.add(PosixFilePermission.OWNER_EXECUTE);
        perms.add(PosixFilePermission.GROUP_READ);
        perms.add(PosixFilePermission.GROUP_EXECUTE);
        perms.add(PosixFilePermission.OTHERS_READ);
        perms.add(PosixFilePermission.OTHERS_EXECUTE);
        Files.setPosixFilePermissions(new File(targetPath).toPath(), perms);
    }

    @NonNull
    private static String extract(@NonNull String lib, @NonNull String resDirPath, @NonNull File targetDir) throws IOException {
        if (!targetDir.exists() && !targetDir.mkdirs()) {
            throw new IOException("Cannot create target directory: " + targetDir.getCanonicalPath());
        }
        File targetFile = new File(targetDir, lib);
        String targetPath = targetFile.getCanonicalPath();
        if (targetFile.exists()) {
            return targetPath;
        }
        String resPath = resDirPath + JAVA_PATH_SEPARATOR + lib;
        try (InputStream in = NativeLibrary.class.getResourceAsStream(resPath);){
            if (in == null) {
                throw new IOException("Cannot find resource for native library at " + resPath);
            }
            try (OutputStream out = Files.newOutputStream(targetFile.toPath(), new OpenOption[0]);){
                int bytesRead;
                byte[] buf = new byte[1024];
                while ((bytesRead = in.read(buf)) != -1) {
                    out.write(buf, 0, bytesRead);
                }
            }
        }
        return targetPath;
    }

    private static boolean isWindows(@NonNull String sysOs) {
        return sysOs.toLowerCase(Locale.getDefault()).contains("windows");
    }

    private static boolean isMacOS(@NonNull String sysOs) {
        return sysOs.toLowerCase(Locale.getDefault()).contains(OS_MAC);
    }

    private static boolean isLinux(@NonNull String sysOs) {
        return sysOs.toLowerCase(Locale.getDefault()).contains("linux");
    }

    @NonNull
    private static String getRootPath(@NonNull String os) {
        return "/libs/" + NativeLibrary.getOsDir(os) + JAVA_PATH_SEPARATOR;
    }
}

