"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const graphql_1 = require("graphql");
const builtInScalarTypes = new Set([graphql_1.GraphQLString, graphql_1.GraphQLInt, graphql_1.GraphQLFloat, graphql_1.GraphQLBoolean, graphql_1.GraphQLID]);
function isBuiltInScalarType(type) {
    return builtInScalarTypes.has(type);
}
exports.isBuiltInScalarType = isBuiltInScalarType;
const typenameField = { kind: graphql_1.Kind.FIELD, name: { kind: graphql_1.Kind.NAME, value: '__typename' } };
function withTypenameFieldAddedWhereNeeded(schema, ast) {
    function isOperationRootType(type) {
        return type === schema.getQueryType() ||
            type === schema.getMutationType() ||
            type === schema.getSubscriptionType();
    }
    const typeInfo = new graphql_1.TypeInfo(schema);
    return graphql_1.visit(ast, graphql_1.visitWithTypeInfo(typeInfo, {
        leave: {
            SelectionSet: (node) => {
                const parentType = typeInfo.getParentType();
                if (!isOperationRootType(parentType)) {
                    return Object.assign({}, node, { selections: [typenameField, ...node.selections] });
                }
                else {
                    return undefined;
                }
            }
        }
    }));
}
exports.withTypenameFieldAddedWhereNeeded = withTypenameFieldAddedWhereNeeded;
function sourceAt(location) {
    return location.source.body.slice(location.start, location.end);
}
exports.sourceAt = sourceAt;
function filePathForNode(node) {
    const name = node.loc && node.loc.source && node.loc.source.name;
    return (name === "GraphQL") ? undefined : name;
}
exports.filePathForNode = filePathForNode;
function valueFromValueNode(valueNode) {
    switch (valueNode.kind) {
        case 'IntValue':
        case 'FloatValue':
            return Number(valueNode.value);
        case 'NullValue':
            return null;
        case 'ListValue':
            return valueNode.values.map(valueFromValueNode);
        case 'ObjectValue':
            return valueNode.fields.reduce((object, field) => {
                object[field.name.value] = valueFromValueNode(field.value);
                return object;
            }, {});
        case 'Variable':
            return { kind: 'Variable', variableName: valueNode.name.value };
        default:
            return valueNode.value;
    }
}
exports.valueFromValueNode = valueFromValueNode;
function isTypeProperSuperTypeOf(schema, maybeSuperType, subType) {
    return graphql_1.isEqualType(maybeSuperType, subType) || subType instanceof graphql_1.GraphQLObjectType && (graphql_1.isAbstractType(maybeSuperType) && schema.isPossibleType(maybeSuperType, subType));
}
exports.isTypeProperSuperTypeOf = isTypeProperSuperTypeOf;
function getOperationRootType(schema, operation) {
    switch (operation.operation) {
        case 'query':
            return schema.getQueryType();
        case 'mutation':
            const mutationType = schema.getMutationType();
            if (!mutationType) {
                throw new graphql_1.GraphQLError('Schema is not configured for mutations', [operation]);
            }
            return mutationType;
        case 'subscription':
            const subscriptionType = schema.getSubscriptionType();
            if (!subscriptionType) {
                throw new graphql_1.GraphQLError('Schema is not configured for subscriptions', [operation]);
            }
            return subscriptionType;
        default:
            throw new graphql_1.GraphQLError('Can only compile queries, mutations and subscriptions', [operation]);
    }
}
exports.getOperationRootType = getOperationRootType;
function getFieldDef(schema, parentType, fieldAST) {
    const name = fieldAST.name.value;
    if (name === graphql_1.SchemaMetaFieldDef.name &&
        schema.getQueryType() === parentType) {
        return graphql_1.SchemaMetaFieldDef;
    }
    if (name === graphql_1.TypeMetaFieldDef.name &&
        schema.getQueryType() === parentType) {
        return graphql_1.TypeMetaFieldDef;
    }
    if (name === graphql_1.TypeNameMetaFieldDef.name &&
        (parentType instanceof graphql_1.GraphQLObjectType ||
            parentType instanceof graphql_1.GraphQLInterfaceType ||
            parentType instanceof graphql_1.GraphQLUnionType)) {
        return graphql_1.TypeNameMetaFieldDef;
    }
    if (parentType instanceof graphql_1.GraphQLObjectType ||
        parentType instanceof graphql_1.GraphQLInterfaceType) {
        return parentType.getFields()[name];
    }
    return undefined;
}
exports.getFieldDef = getFieldDef;
//# sourceMappingURL=graphql.js.map