/**
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.crosstreelabs.phpfunctions;

import com.crosstreelabs.phpfunctions.dateformat.LowercaseHalfOfDayPrinter;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.joda.time.DateTime;
import org.joda.time.DateTimeFieldType;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;
import org.joda.time.format.DateTimeParser;
import org.joda.time.format.DateTimeParserBucket;

public class DateParsingUtil {
    
    public static final DateTimeFormatter TIMEZONE = new DateTimeFormatterBuilder()
            .appendOptional(new OneOf("UTC", "utc", "GMT", "gmt"))
            .append(DateTimeFormat.forPattern("ZZ"))
            .toFormatter();
    
    /**
     * The formats for parsing 12-hour times are derived from the notations
     * given at: http://php.net/manual/en/datetime.formats.time.php
     */
    public static final DateTimeFormatter TWELVE_HOUR_TIME = new DateTimeFormatterBuilder()
            .appendClockhourOfHalfday(1)
            .appendOptional(new DateTimeFormatterBuilder()
                    .appendOptional(new OneOf(".",":"))
                    .appendMinuteOfHour(2)
                    .appendOptional(new DateTimeFormatterBuilder()
                            .appendOptional(new OneOf(".",":"))
                            .appendSecondOfMinute(2)
                            .appendOptional(new DateTimeFormatterBuilder()
                                    .appendOptional(new OneOf(".",":"))
                                    .appendFractionOfSecond(1, 6)
                                    .toParser()
                            )
                            .toParser()
                    )
                    .toParser()
            )
            .appendOptional(DateTimeFormat.forPattern(" ").getParser())
            .append(new LowercaseHalfOfDayPrinter(), new LowercaseHalfOfDayPrinter())
            .toFormatter()
            .withOffsetParsed();
    
    /**
     * The formats for parsing 24-hour times are derived from the notations
     * given at: http://php.net/manual/en/datetime.formats.time.php
     */
    public static final DateTimeFormatter TWENTY_FOUR_HOUR_TIME = new DateTimeFormatterBuilder()
            .append(null, new DateTimeParser[]{
                TIMEZONE.getParser(),
                new DateTimeFormatterBuilder()
                        .appendOptional(new OneOf("t","T"))
                        .appendHourOfDay(2)
                        .appendOptional(new OneOf(".", ":"))
                        .appendMinuteOfHour(2)
                        .appendOptional(new DateTimeFormatterBuilder()
                                .appendOptional(new OneOf(".", ":"))
                                .appendSecondOfMinute(2)
                                .appendOptional(new DateTimeFormatterBuilder()
                                        .append(null, new DateTimeParser[]{
                                            TIMEZONE.getParser(),
                                            DateTimeFormat.forPattern(".SSSSSS").getParser()
                                        })
                                        .toParser()
                                )
                                .toParser()
                        ).toParser()
            }).toFormatter()
            .withOffsetParsed();
    
    /**
     * The formats for parsing general dates are derived from the notations
     * given at: http://php.net/manual/en/datetime.formats.date.php
     */
    public static final DateTimeFormatter DATE_FORMATS = new DateTimeFormatterBuilder()
            .append(null, new DateTimeParser[]{
                // American style
                new DateTimeFormatterBuilder()
                    .appendMonthOfYear(1)
                    .appendLiteral('/')
                    .appendDayOfMonth(1)
                    .appendOptional(new DateTimeFormatterBuilder()
                            .appendLiteral('/')
                            .append(new YearParser())
                            .toParser()
                    )
                    .toParser(),
                DateTimeFormat.forPattern("yyyy/M/d").getParser(),
                DateTimeFormat.forPattern("yyyy-M").getParser(),
                new DateTimeFormatterBuilder()
                    .append(new YearParser())
                    .appendLiteral('-')
                    .appendMonthOfYear(1)
                    .appendLiteral('-')
                    .appendDayOfMonth(1)
                    .toParser(),
                new DateTimeFormatterBuilder()
                    .appendDayOfMonth(1)
                    .append(new OneOf("\t", ".", "-"))
                    .appendMonthOfYear(1)
                    .append(new OneOf("\t", ".", "-"))
                    .append(new YearParser(2, 4))
                    .toParser(),
                new DateTimeFormatterBuilder()
                    .append(new DayParser())
                    .append(new ManyOf(" ", "\t", ".", "-"))
                    .appendPattern("MMMM")
                    .append(new ManyOf(" ", "\t", ".", "-"))
                    .append(new YearParser())
                    .toParser(),
                new DateTimeFormatterBuilder() // Textual month and four digit year (Day reset to 1)
                    .appendPattern("MMMM")
                    .append(new ManyOf(" ", "\t", ".", "-"))
                    .append(new YearParser(4, 4))
                    .toParser(),
                new DateTimeFormatterBuilder()
                    .append(new YearParser(4, 4))
                    .append(new ManyOf(" ", "\t", ".", "-"))
                    .appendPattern("MMMM")
                    .toParser(),
                new DateTimeFormatterBuilder() // Textual month, day (optional year)
                    .appendPattern("MMMM")
                    .append(new ManyOf(" ", "\t", ".", "-"))
                    .appendDayOfMonth(1)
                    .append(new ManyOf(",",".","s","t","n","d","r","h","\t"," "))
                    .appendOptional(new YearParser())
                    .toParser(),
                new DateTimeFormatterBuilder() // Day and textual month
                    .appendDayOfMonth(1)
                    .append(new ManyOf(" ", "\t", ".", "-"))
                    .appendPattern("MMMM")
                    .toParser(),
                new DateTimeFormatterBuilder() // Month abbreviation, day and year
                    .appendPattern("MMM")
                    .appendLiteral('-')
                    .appendDayOfMonth(2)
                    .appendLiteral('-')
                    .append(new YearParser())
                    .toParser(),
                new DateTimeFormatterBuilder() // Year, month abbreviation, and day
                    .append(new YearParser())
                    .appendLiteral('-')
                    .appendPattern("MMM")
                    .appendLiteral('-')
                    .appendDayOfMonth(2)
                    .toParser(),
                DateTimeFormat.forPattern("yyyy").getParser(),
                DateTimeFormat.forPattern("MMM").getParser()
            })
            .toFormatter()
            .withDefaultYear(Calendar.getInstance().get(Calendar.YEAR))
            .withOffsetParsed();
    
    /**
     * The formats for parsing ISO dates are derived from the notations
     * given at: http://php.net/manual/en/datetime.formats.date.php
     */
    public static final DateTimeFormatter ISO_DATE_FORMATS = new DateTimeFormatterBuilder()
            .append(null, new DateTimeParser[]{
                DateTimeFormat.forPattern("yyyyMMdd").getParser(),
                DateTimeFormat.forPattern("yyyy/MM/dd").getParser(),
                DateTimeFormat.forPattern("yy-MM-dd").getParser(),
                new DateTimeFormatterBuilder()
                    .append(new YearParser(4, 4, true))
                    .append(DateTimeFormat.forPattern("-MM-dd"))
                    .toParser()
            })
            .toFormatter()
            .withOffsetParsed();
    
    /**
     * The formats for parsing compound datetimes are derived from the notations
     * given at: http://php.net/manual/en/datetime.formats.compound.php
     */
    public static final DateTimeFormatter COMPOUND_FORMATS = new DateTimeFormatterBuilder()
            .append(null, new DateTimeParser[]{
                DateTimeFormat.forPattern("dd/MMM/yyyy:HH:mm:ss ZZ").getParser(),
                DateTimeFormat.forPattern("yyyy:MM:dd HH:mm:ss").getParser(),
                new DateTimeFormatterBuilder() // IOS year with ISO week (optional day)
                    .appendYear(4, 4)
                    .appendOptional(new OneOf("-"))
                    .appendLiteral('W')
                    .appendWeekOfWeekyear(2)
                    .appendOptional(new DateTimeFormatterBuilder()
                            .appendOptional(new OneOf("-"))
                            .appendDayOfWeek(1)
                            .toParser()
                    )
                    .toParser(),
                DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss").getParser(),
                new DateTimeFormatterBuilder()
                    .appendYear(4, 4)
                    .appendOptional(new OneOf("."))
                    .appendDayOfYear(3)
                    .toParser(),
                DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSSSSZZ").getParser(),
                DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSSSS").getParser(),
                new DateTimeFormatterBuilder()
                    .appendLiteral('@')
                    .append(new UnixTimestampParser())
                    .toParser(),
                DateTimeFormat.forPattern("yyyyMMdd'T'HH:mm:ss").getParser(),
                DateTimeFormat.forPattern("yyyyMMdd't'HHmmss").getParser(),
                DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss").getParser()
            })
            .toFormatter()
            .withOffsetParsed();
    
    
    public static class OneOf implements DateTimeParser {
        protected final String[] strings;
        protected final int maxLength;
        
        public OneOf(String...strings) {
            this.strings = order(strings);
            this.maxLength = expectedMaxLength();
        }
        
        @Override
        public int estimateParsedLength() {
            return maxLength;
        }

        @Override
        public int parseInto(DateTimeParserBucket bucket, String text, int position) {
            String next = text.substring(position);
            int longest = 0;
            for (String str : strings) {
                if (next.startsWith(str) && str.length() > longest) {
                    longest = str.length();
                }
            }
            return position + longest;
        }
        
        protected String[] order(String[] strings) {
            List<String> list = Arrays.asList(strings);
            Collections.sort(list, new Comparator<String>() {
                @Override
                public int compare(String t, String t1) {
                    return t.length() - t1.length();
                }
            });
            return list.toArray(new String[list.size()]);
        }
        protected int expectedMaxLength() {
            int maxLength = 0;
            for (String str : strings) {
                if (str.length() > maxLength) {
                    maxLength = str.length();
                }
            }
            return maxLength;
        }
    }
    public static class ManyOf extends OneOf {
        public ManyOf(String...strings) {
            super(strings);
        }
        
        @Override
        public int parseInto(DateTimeParserBucket bucket, String text, int position) {
            String next = text.substring(position);
            while (true) {
                boolean started = false;
                for (String str : strings) {
                    if (next.startsWith(str)) {
                        started = true;
                        next = next.substring(str.length());
                        position += str.length();
                    }
                }
                if (!started) {
                    break;
                }
            }
            return position;
        }
    }
    
    public static class YearParser implements DateTimeParser {
        private final int minLength;
        private final int maxLength;
        private final boolean allowLeadingSign;
        
        public YearParser() {
            this(1, 4, false);
        }
        public YearParser(final int minLength, final int maxLength) {
            this(minLength, maxLength, false);
        }
        public YearParser(final int minLength, final int maxLength, final boolean allowLeadingSign) {
            this.minLength = minLength;
            this.maxLength = maxLength;
            this.allowLeadingSign = allowLeadingSign;
        }
        
        @Override
        public int estimateParsedLength() {
            return maxLength;
        }

        @Override
        public int parseInto(DateTimeParserBucket bucket, String text, int position) {
            String next = text.substring(position);
            String number = getFirstNumber(next);
            if (number == null || number.length() < minLength
                    || number.length() > maxLength) {
                return -1;
            }
            Integer year = Integer.valueOf(number);
            if (number.length() >= 4) {
                bucket.saveField(DateTimeFieldType.year(), year);
                return position+number.length();
            }
            
            int pivot = (bucket.getPivotYear() == null ? 69 : bucket.getPivotYear()) % 100;
            if (year >= 0 && year <= pivot) {
                bucket.saveField(DateTimeFieldType.year(), 2000 + year);
            } else {
                bucket.saveField(DateTimeFieldType.year(), 1900 + year);
            }
            return position+number.length();
        }
        
        protected String getFirstNumber(String text) {
            Pattern p = Pattern.compile(String.format("^%s\\d{%s,%s}", (allowLeadingSign ? "[-+]?" : "" ), minLength, maxLength));
            Matcher m = p.matcher(text);
            if (m.find()) {
                return m.group();
            }
            return null;
        }
    }
    public static class DayParser implements DateTimeParser {

        @Override
        public int estimateParsedLength() {
            return 2;
        }

        @Override
        public int parseInto(DateTimeParserBucket bucket, String text, int position) {
            String next = text.substring(position);
            String number = getFirstNumber(next);
            if (number == null) {
                return -1;
            }
            Integer day = Integer.valueOf(number);
            if (day < 0 || day > 31) {
                return -1;
            }
            bucket.saveField(DateTimeFieldType.dayOfMonth(), day);
            return position+number.length();
        }
        protected String getFirstNumber(String text) {
            Pattern p = Pattern.compile("^\\d{1,2}");
            Matcher m = p.matcher(text);
            if (m.find()) {
                return m.group();
            }
            return null;
        }
    }
    public static class UnixTimestampParser implements DateTimeParser {

        @Override
        public int estimateParsedLength() {
            return 20; // Java uses longs for timestamp representation
        }

        @Override
        public int parseInto(DateTimeParserBucket bucket, String text, int position) {
            String next = text.substring(position);
            Pattern p = Pattern.compile("^-?\\d{1,20}");
            Matcher m = p.matcher(next);
            if (!m.find()) {
                return 0;
            }
            String timestampStr = m.group();
            if (timestampStr == null || timestampStr.trim().isEmpty()) {
                return 0;
            }
            Long timestamp = Long.valueOf(timestampStr);
            DateTime date = new DateTime(timestamp * 1000, DateTimeZone.UTC);
            
            
            bucket.setZone(DateTimeZone.UTC);
            bucket.saveField(DateTimeFieldType.year(), date.getYear());
            bucket.saveField(DateTimeFieldType.monthOfYear(), date.getMonthOfYear());
            bucket.saveField(DateTimeFieldType.dayOfMonth(), date.getDayOfMonth());
            bucket.saveField(DateTimeFieldType.hourOfDay(), date.getHourOfDay());
            bucket.saveField(DateTimeFieldType.minuteOfHour(), date.getMinuteOfHour());
            bucket.saveField(DateTimeFieldType.secondOfMinute(), date.getSecondOfMinute());
            bucket.saveField(DateTimeFieldType.millisOfSecond(), date.getMillisOfSecond());
            return position+timestampStr.length();
        }
        
    }
}