/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.sdk.core;

import com.databricks.sdk.core.BodyLogger;
import com.databricks.sdk.core.ConfigLoader;
import com.databricks.sdk.core.DatabricksConfig;
import com.databricks.sdk.core.DatabricksError;
import com.databricks.sdk.core.DatabricksException;
import com.databricks.sdk.core.GrpcTranscodingQueryParamsSerializer;
import com.databricks.sdk.core.UserAgent;
import com.databricks.sdk.core.error.ApiErrors;
import com.databricks.sdk.core.http.HttpClient;
import com.databricks.sdk.core.http.Request;
import com.databricks.sdk.core.http.Response;
import com.databricks.sdk.core.utils.SerDeUtils;
import com.databricks.sdk.core.utils.SystemTimer;
import com.databricks.sdk.core.utils.Timer;
import com.databricks.sdk.support.Header;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApiClient {
    private static final Logger LOG = LoggerFactory.getLogger(ApiClient.class);
    private final int maxAttempts;
    private final ObjectMapper mapper;
    private final DatabricksConfig config;
    private final Random random;
    private final HttpClient httpClient;
    private final BodyLogger bodyLogger;
    private final Timer timer;

    public ApiClient() {
        this(ConfigLoader.getDefault());
    }

    public String configuredAccountID() {
        return this.config.getAccountId();
    }

    public ApiClient(DatabricksConfig config) {
        this(config, new SystemTimer());
    }

    public ApiClient(DatabricksConfig config, Timer timer) {
        Integer debugTruncateBytes;
        this.config = config;
        config.resolve();
        Integer rateLimit = config.getRateLimit();
        if (rateLimit == null) {
            rateLimit = 15;
        }
        if ((debugTruncateBytes = config.getDebugTruncateBytes()) == null) {
            debugTruncateBytes = 96;
        }
        this.maxAttempts = 3;
        this.mapper = SerDeUtils.createMapper();
        this.random = new Random();
        this.httpClient = config.getHttpClient();
        this.bodyLogger = new BodyLogger(this.mapper, 1024, debugTruncateBytes);
        this.timer = timer;
    }

    private static <I> void setQuery(Request in, I entity) {
        if (entity == null) {
            return;
        }
        for (GrpcTranscodingQueryParamsSerializer.QueryParamPair e : GrpcTranscodingQueryParamsSerializer.serialize(entity)) {
            in.withQueryParam(e.getKey(), e.getValue());
        }
    }

    private static <I> void setHeaders(Request in, Map<String, String> headers) {
        if (headers == null) {
            return;
        }
        for (Map.Entry<String, String> e : headers.entrySet()) {
            in.withHeader(e.getKey(), e.getValue());
        }
    }

    public <I, O> Collection<O> getCollection(String path, I in, Class<O> element, Map<String, String> headers) {
        return (Collection)this.withJavaType(path, in, (JavaType)this.mapper.getTypeFactory().constructCollectionType(Collection.class, element), headers);
    }

    public <I> Map<String, String> getStringMap(String path, I in, Map<String, String> headers) {
        return (Map)this.withJavaType(path, in, (JavaType)this.mapper.getTypeFactory().constructMapType(Map.class, String.class, String.class), headers);
    }

    protected <I, O> O withJavaType(String path, I in, JavaType javaType, Map<String, String> headers) {
        try {
            Request request = this.prepareRequest("GET", path, in, headers);
            Response response = this.getResponse(request);
            return (O)this.deserialize(response.getBody(), javaType);
        }
        catch (IOException e) {
            throw new DatabricksException("IO error: " + e.getMessage(), e);
        }
    }

    public <O> O HEAD(String path, Class<O> target, Map<String, String> headers) {
        return this.HEAD(path, null, target, headers);
    }

    public <I, O> O HEAD(String path, I in, Class<O> target, Map<String, String> headers) {
        try {
            return this.execute(this.prepareRequest("HEAD", path, in, headers), target);
        }
        catch (IOException e) {
            throw new DatabricksException("IO error: " + e.getMessage(), e);
        }
    }

    public <O> O GET(String path, Class<O> target, Map<String, String> headers) {
        return this.GET(path, null, target, headers);
    }

    public <I, O> O GET(String path, I in, Class<O> target, Map<String, String> headers) {
        try {
            return this.execute(this.prepareRequest("GET", path, in, headers), target);
        }
        catch (IOException e) {
            throw new DatabricksException("IO error: " + e.getMessage(), e);
        }
    }

    public <I, O> O POST(String path, I in, Class<O> target, Map<String, String> headers) {
        try {
            return this.execute(this.prepareRequest("POST", path, in, headers), target);
        }
        catch (IOException e) {
            throw new DatabricksException("IO error: " + e.getMessage(), e);
        }
    }

    public <I, O> O PUT(String path, I in, Class<O> target, Map<String, String> headers) {
        try {
            return this.execute(this.prepareRequest("PUT", path, in, headers), target);
        }
        catch (IOException e) {
            throw new DatabricksException("IO error: " + e.getMessage(), e);
        }
    }

    public <I, O> O PATCH(String path, I in, Class<O> target, Map<String, String> headers) {
        try {
            return this.execute(this.prepareRequest("PATCH", path, in, headers), target);
        }
        catch (IOException e) {
            throw new DatabricksException("IO error: " + e.getMessage(), e);
        }
    }

    public <I, O> O DELETE(String path, I in, Class<O> target, Map<String, String> headers) {
        try {
            return this.execute(this.prepareRequest("DELETE", path, in, headers), target);
        }
        catch (IOException e) {
            throw new DatabricksException("IO error: " + e.getMessage(), e);
        }
    }

    private boolean hasBody(String method) {
        return !method.equals("GET") && !method.equals("DELETE") && !method.equals("HEAD");
    }

    private <I> Request prepareBaseRequest(String method, String path, I in) throws JsonProcessingException {
        if (in == null || !this.hasBody(method)) {
            return new Request(method, path);
        }
        if (InputStream.class.isAssignableFrom(in.getClass())) {
            InputStream body = (InputStream)in;
            return new Request(method, path, body);
        }
        String body = this.serialize(in);
        return new Request(method, path, body);
    }

    private <I> Request prepareRequest(String method, String path, I in, Map<String, String> headers) throws JsonProcessingException {
        Request req = this.prepareBaseRequest(method, path, in);
        ApiClient.setQuery(req, in);
        ApiClient.setHeaders(req, headers);
        return req;
    }

    private <T> T execute(Request in, Class<T> target) throws IOException {
        Response out = this.getResponse(in);
        if (target == Void.class) {
            return null;
        }
        return this.deserialize(out, target);
    }

    private Response getResponse(Request in) {
        in.withUrl(this.config.getHost() + in.getUrl());
        return this.executeInner(in);
    }

    private Response executeInner(Request in) {
        int attemptNumber = 0;
        while (true) {
            ++attemptNumber;
            IOException err = null;
            Response out = null;
            in.withHeaders(this.config.authenticate());
            String userAgent = String.format("%s auth/%s", UserAgent.asString(), this.config.getAuthType());
            in.withHeader("User-Agent", userAgent);
            try {
                out = this.httpClient.execute(in);
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.makeLogRecord(in, out));
                }
                if (out.getStatusCode() < 400) {
                    return out;
                }
            }
            catch (IOException e) {
                err = e;
                LOG.debug("Request {} failed", (Object)in, (Object)e);
            }
            DatabricksError res = ApiErrors.checkForRetry(out, err);
            if (!res.isRetriable()) {
                if (res.getErrorCode() == null) {
                    return out;
                }
                throw res;
            }
            if (attemptNumber == this.maxAttempts) {
                throw new DatabricksException(String.format("Request %s failed after %d retries", in, this.maxAttempts), err);
            }
            int sleepMillis = this.getBackoffMillis(attemptNumber);
            LOG.debug(String.format("Retry %s in %dms", in.getRequestLine(), sleepMillis));
            try {
                this.timer.wait(sleepMillis);
                continue;
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                continue;
            }
            break;
        }
    }

    private int getBackoffMillis(int attemptNumber) {
        int maxWait = 10000;
        int minJitter = 50;
        int maxJitter = 750;
        int wait = Math.min(maxWait, attemptNumber * 1000);
        return wait += this.random.nextInt(maxJitter - minJitter + 1) + minJitter;
    }

    private String makeLogRecord(Request in, Response out) {
        StringBuilder sb = new StringBuilder();
        sb.append("> ");
        sb.append(in.getRequestLine());
        if (this.config.isDebugHeaders()) {
            sb.append("\n * Host: ");
            sb.append(this.config.getHost());
            in.getHeaders().forEach((header, value) -> sb.append(String.format("\n * %s: %s", header, value)));
        }
        if (in.isBodyStreaming()) {
            sb.append("\n> (streamed body)");
        } else {
            String requestBody = in.getBodyString();
            if (requestBody != null && !requestBody.isEmpty()) {
                String[] stringArray = this.bodyLogger.redactedDump(requestBody).split("\n");
                int n = stringArray.length;
                for (int i = 0; i < n; ++i) {
                    String line = stringArray[i];
                    sb.append("\n> ");
                    sb.append(line);
                }
            }
        }
        sb.append("\n< ");
        sb.append(out.toString());
        for (String line : this.bodyLogger.redactedDump(out.getDebugBody()).split("\n")) {
            sb.append("\n< ");
            sb.append(line);
        }
        return sb.toString();
    }

    public <T> T deserialize(Response response, Class<T> target) throws IOException {
        T object;
        if (target == InputStream.class) {
            return (T)response.getBody();
        }
        try {
            object = target.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (Exception e) {
            throw new DatabricksException("Unable to initialize an instance of type " + target.getName());
        }
        this.deserialize(response, object);
        return object;
    }

    public <T> T deserialize(InputStream body, JavaType target) throws IOException {
        if (target == this.mapper.constructType(InputStream.class)) {
            return (T)body;
        }
        return (T)this.mapper.readValue(body, target);
    }

    private <T> void fillInHeaders(T target, Response response) {
        for (Field field : target.getClass().getDeclaredFields()) {
            String firstHeader;
            Header headerAnnotation = field.getAnnotation(Header.class);
            if (headerAnnotation == null || (firstHeader = response.getFirstHeader(headerAnnotation.value())) == null) continue;
            try {
                field.setAccessible(true);
                if (field.getType() == String.class) {
                    field.set(target, firstHeader);
                    continue;
                }
                if (field.getType() == Long.class) {
                    field.set(target, Long.parseLong(firstHeader));
                    continue;
                }
                LOG.warn("Unsupported header type: " + field.getType());
            }
            catch (IllegalAccessException e) {
                throw new DatabricksException("Failed to unmarshal headers: " + e.getMessage(), e);
            }
            finally {
                field.setAccessible(false);
            }
        }
    }

    private <T> Optional<Field> getContentsField(T target) {
        for (Field field : target.getClass().getDeclaredFields()) {
            if (!field.getName().equals("contents") || field.getType() != InputStream.class) continue;
            return Optional.of(field);
        }
        return Optional.empty();
    }

    public <T> void deserialize(Response response, T object) throws IOException {
        this.fillInHeaders(object, response);
        Optional<Field> contentsField = this.getContentsField(object);
        if (contentsField.isPresent()) {
            Field field = contentsField.get();
            try {
                field.setAccessible(true);
                field.set(object, response.getBody());
            }
            catch (IllegalAccessException e) {
                throw new DatabricksException("Failed to unmarshal headers: " + e.getMessage(), e);
            }
            finally {
                field.setAccessible(false);
            }
        } else if (response.getBody() != null) {
            this.mapper.readerForUpdating(object).readValue(response.getBody());
        }
    }

    private String serialize(Object body) throws JsonProcessingException {
        if (body == null) {
            return null;
        }
        return this.mapper.writeValueAsString(body);
    }
}

