/*
 * Decompiled with CFR 0.152.
 */
package com.databricks.sdk.core.oauth;

import com.databricks.sdk.core.ApiClient;
import com.databricks.sdk.core.DatabricksException;
import com.databricks.sdk.core.http.FormRequest;
import com.databricks.sdk.core.http.HttpClient;
import com.databricks.sdk.core.http.Request;
import com.databricks.sdk.core.oauth.AuthParameterPosition;
import com.databricks.sdk.core.oauth.OAuthResponse;
import com.databricks.sdk.core.oauth.Token;
import com.databricks.sdk.core.oauth.TokenSource;
import com.databricks.sdk.core.utils.ClockSupplier;
import com.databricks.sdk.core.utils.UtcClockSupplier;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class RefreshableTokenSource
implements TokenSource {
    private static final Logger logger = LoggerFactory.getLogger(RefreshableTokenSource.class);
    private static final Duration DEFAULT_STALE_DURATION = Duration.ofMinutes(3L);
    private static final Duration DEFAULT_EXPIRY_BUFFER = Duration.ofSeconds(40L);
    protected volatile Token token;
    private boolean asyncEnabled = Boolean.parseBoolean(System.getenv("DATABRICKS_ENABLE_EXPERIMENTAL_ASYNC_TOKEN_REFRESH"));
    private Duration staleDuration = DEFAULT_STALE_DURATION;
    private Duration expiryBuffer = DEFAULT_EXPIRY_BUFFER;
    private boolean refreshInProgress = false;
    private boolean lastRefreshSucceeded = true;
    private ClockSupplier clockSupplier = new UtcClockSupplier();

    public RefreshableTokenSource() {
    }

    public RefreshableTokenSource(Token token) {
        this.token = token;
    }

    public RefreshableTokenSource withClockSupplier(ClockSupplier clockSupplier) {
        this.clockSupplier = clockSupplier;
        return this;
    }

    public RefreshableTokenSource withAsyncRefresh(boolean enabled) {
        this.asyncEnabled = enabled;
        return this;
    }

    public RefreshableTokenSource withExpiryBuffer(Duration buffer) {
        this.expiryBuffer = buffer;
        return this;
    }

    protected abstract Token refresh();

    @Override
    public Token getToken() {
        if (this.asyncEnabled) {
            return this.getTokenAsync();
        }
        return this.getTokenBlocking();
    }

    protected TokenState getTokenState(Token t) {
        if (t == null) {
            return TokenState.EXPIRED;
        }
        Duration lifeTime = Duration.between(Instant.now(this.clockSupplier.getClock()), t.getExpiry());
        if (lifeTime.compareTo(this.expiryBuffer) <= 0) {
            return TokenState.EXPIRED;
        }
        if (lifeTime.compareTo(this.staleDuration) <= 0) {
            return TokenState.STALE;
        }
        return TokenState.FRESH;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected Token getTokenBlocking() {
        if (this.getTokenState(this.token) != TokenState.EXPIRED) {
            return this.token;
        }
        RefreshableTokenSource refreshableTokenSource = this;
        synchronized (refreshableTokenSource) {
            if (this.getTokenState(this.token) != TokenState.EXPIRED) {
                return this.token;
            }
            this.lastRefreshSucceeded = false;
            try {
                this.token = this.refresh();
            }
            catch (Exception e) {
                logger.error("Failed to refresh token synchronously", (Throwable)e);
                throw e;
            }
            this.lastRefreshSucceeded = true;
            return this.token;
        }
    }

    protected Token getTokenAsync() {
        Token currentToken = this.token;
        switch (this.getTokenState(currentToken)) {
            case FRESH: {
                return currentToken;
            }
            case STALE: {
                this.triggerAsyncRefresh();
                return currentToken;
            }
            case EXPIRED: {
                return this.getTokenBlocking();
            }
        }
        throw new IllegalStateException("Invalid token state.");
    }

    private synchronized void triggerAsyncRefresh() {
        if (!this.refreshInProgress && this.lastRefreshSucceeded && this.getTokenState(this.token) != TokenState.FRESH) {
            this.refreshInProgress = true;
            CompletableFuture.runAsync(() -> {
                try {
                    Token newToken = this.refresh();
                    RefreshableTokenSource refreshableTokenSource = this;
                    synchronized (refreshableTokenSource) {
                        this.token = newToken;
                        this.refreshInProgress = false;
                    }
                }
                catch (Exception e) {
                    RefreshableTokenSource refreshableTokenSource = this;
                    synchronized (refreshableTokenSource) {
                        this.lastRefreshSucceeded = false;
                        this.refreshInProgress = false;
                        logger.error("Asynchronous token refresh failed", (Throwable)e);
                    }
                }
            });
        }
    }

    protected static Token retrieveToken(HttpClient hc, String clientId, String clientSecret, String tokenUrl, Map<String, String> params, Map<String, String> headers, AuthParameterPosition position) {
        switch (position) {
            case BODY: {
                if (clientId != null) {
                    params.put("client_id", clientId);
                }
                if (clientSecret == null) break;
                params.put("client_secret", clientSecret);
                break;
            }
            case HEADER: {
                String authHeaderValue = "Basic " + Base64.getEncoder().encodeToString((clientId + ":" + clientSecret).getBytes());
                headers.put("Authorization", authHeaderValue);
            }
        }
        headers.put("Content-Type", "application/x-www-form-urlencoded");
        Request req = new Request("POST", tokenUrl, FormRequest.wrapValuesInList(params));
        req.withHeaders(headers);
        try {
            ApiClient apiClient = new ApiClient.Builder().withHttpClient(hc).build();
            OAuthResponse resp = apiClient.execute(req, OAuthResponse.class);
            if (resp.getErrorCode() != null) {
                throw new IllegalArgumentException(resp.getErrorCode() + ": " + resp.getErrorSummary());
            }
            Instant expiry = Instant.now().plusSeconds(resp.getExpiresIn());
            return new Token(resp.getAccessToken(), resp.getTokenType(), resp.getRefreshToken(), expiry);
        }
        catch (Exception e) {
            throw new DatabricksException("Failed to refresh credentials: " + e.getMessage(), e);
        }
    }

    protected static enum TokenState {
        FRESH,
        STALE,
        EXPIRED;

    }
}

