package com.datadog.android.telemetry.model

import com.datadog.android.core.`internal`.utils.JsonSerializer
import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Any
import kotlin.Array
import kotlin.Long
import kotlin.Number
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.collections.MutableMap
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Schema of common properties of Telemetry events
 * @param dd Internal properties
 * @param date Start of the event in ms from epoch
 * @param service The SDK generating the telemetry event
 * @param source The source of this event
 * @param version The version of the SDK generating the telemetry event
 * @param application Application properties
 * @param session Session properties
 * @param view View properties
 * @param action Action properties
 * @param effectiveSampleRate The actual percentage of telemetry usage per event
 * @param experimentalFeatures Enabled experimental features
 * @param telemetry The telemetry log information
 */
public data class TelemetryErrorEvent(
    public val dd: Dd,
    public val date: Long,
    public val service: String,
    public val source: Source,
    public val version: String,
    public val application: Application? = null,
    public val session: Session? = null,
    public val view: View? = null,
    public val action: Action? = null,
    public val effectiveSampleRate: Number? = null,
    public val experimentalFeatures: List<String>? = null,
    public val telemetry: Telemetry,
) {
    public val type: String = "telemetry"

    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.add("_dd", dd.toJson())
        json.addProperty("type", type)
        json.addProperty("date", date)
        json.addProperty("service", service)
        json.add("source", source.toJson())
        json.addProperty("version", version)
        application?.let { applicationNonNull ->
            json.add("application", applicationNonNull.toJson())
        }
        session?.let { sessionNonNull ->
            json.add("session", sessionNonNull.toJson())
        }
        view?.let { viewNonNull ->
            json.add("view", viewNonNull.toJson())
        }
        action?.let { actionNonNull ->
            json.add("action", actionNonNull.toJson())
        }
        effectiveSampleRate?.let { effectiveSampleRateNonNull ->
            json.addProperty("effective_sample_rate", effectiveSampleRateNonNull)
        }
        experimentalFeatures?.let { experimentalFeaturesNonNull ->
            val experimentalFeaturesArray = JsonArray(experimentalFeaturesNonNull.size)
            experimentalFeaturesNonNull.forEach { experimentalFeaturesArray.add(it) }
            json.add("experimental_features", experimentalFeaturesArray)
        }
        json.add("telemetry", telemetry.toJson())
        return json
    }

    public companion object {
        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): TelemetryErrorEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryErrorEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): TelemetryErrorEvent {
            try {
                val dd = jsonObject.get("_dd").asJsonObject.let {
                    Dd.fromJsonObject(it)
                }
                val type = jsonObject.get("type").asString
                val date = jsonObject.get("date").asLong
                val service = jsonObject.get("service").asString
                val source = Source.fromJson(jsonObject.get("source").asString)
                val version = jsonObject.get("version").asString
                val application = jsonObject.get("application")?.asJsonObject?.let {
                    Application.fromJsonObject(it)
                }
                val session = jsonObject.get("session")?.asJsonObject?.let {
                    Session.fromJsonObject(it)
                }
                val view = jsonObject.get("view")?.asJsonObject?.let {
                    View.fromJsonObject(it)
                }
                val action = jsonObject.get("action")?.asJsonObject?.let {
                    Action.fromJsonObject(it)
                }
                val effectiveSampleRate = jsonObject.get("effective_sample_rate")?.asNumber
                val experimentalFeatures =
                        jsonObject.get("experimental_features")?.asJsonArray?.let { jsonArray ->
                    val collection = ArrayList<String>(jsonArray.size())
                    jsonArray.forEach {
                        collection.add(it.asString)
                    }
                    collection
                }
                val telemetry = jsonObject.get("telemetry").asJsonObject.let {
                    Telemetry.fromJsonObject(it)
                }
                check(type == "telemetry")
                return TelemetryErrorEvent(dd, date, service, source, version, application, session,
                        view, action, effectiveSampleRate, experimentalFeatures, telemetry)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryErrorEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryErrorEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type TelemetryErrorEvent",
                    e
                )
            }
        }
    }

    /**
     * Internal properties
     */
    public class Dd {
        public val formatVersion: Long = 2L

        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("format_version", formatVersion)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dd {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dd {
                try {
                    val formatVersion = jsonObject.get("format_version").asLong
                    check(formatVersion == 2.0.toLong())
                    return Dd()
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }
        }
    }

    /**
     * Application properties
     * @param id UUID of the application
     */
    public data class Application(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Application {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Application {
                try {
                    val id = jsonObject.get("id").asString
                    return Application(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Application",
                        e
                    )
                }
            }
        }
    }

    /**
     * Session properties
     * @param id UUID of the session
     */
    public data class Session(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Session {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Session {
                try {
                    val id = jsonObject.get("id").asString
                    return Session(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Session",
                        e
                    )
                }
            }
        }
    }

    /**
     * View properties
     * @param id UUID of the view
     */
    public data class View(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): View {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): View {
                try {
                    val id = jsonObject.get("id").asString
                    return View(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type View",
                        e
                    )
                }
            }
        }
    }

    /**
     * Action properties
     * @param id UUID of the action
     */
    public data class Action(
        public val id: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("id", id)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Action {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Action {
                try {
                    val id = jsonObject.get("id").asString
                    return Action(id)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Action",
                        e
                    )
                }
            }
        }
    }

    /**
     * The telemetry log information
     * @param device Device properties
     * @param os OS properties
     * @param message Body of the log
     * @param error Error properties
     */
    public data class Telemetry(
        public val device: Device? = null,
        public val os: Os? = null,
        public val message: String,
        public val error: Error? = null,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public val type: String = "log"

        public val status: String = "error"

        public fun toJson(): JsonElement {
            val json = JsonObject()
            device?.let { deviceNonNull ->
                json.add("device", deviceNonNull.toJson())
            }
            os?.let { osNonNull ->
                json.add("os", osNonNull.toJson())
            }
            json.addProperty("type", type)
            json.addProperty("status", status)
            json.addProperty("message", message)
            error?.let { errorNonNull ->
                json.add("error", errorNonNull.toJson())
            }
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("device", "os", "type",
                    "status", "message", "error")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Telemetry {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Telemetry {
                try {
                    val device = jsonObject.get("device")?.asJsonObject?.let {
                        Device.fromJsonObject(it)
                    }
                    val os = jsonObject.get("os")?.asJsonObject?.let {
                        Os.fromJsonObject(it)
                    }
                    val type = jsonObject.get("type")?.asString
                    val status = jsonObject.get("status").asString
                    val message = jsonObject.get("message").asString
                    val error = jsonObject.get("error")?.asJsonObject?.let {
                        Error.fromJsonObject(it)
                    }
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    if (type != null) {
                        check(type == "log")
                    }
                    check(status == "error")
                    return Telemetry(device, os, message, error, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Telemetry",
                        e
                    )
                }
            }
        }
    }

    /**
     * Device properties
     * @param architecture Architecture of the device
     * @param brand Brand of the device
     * @param model Model of the device
     */
    public data class Device(
        public val architecture: String? = null,
        public val brand: String? = null,
        public val model: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            architecture?.let { architectureNonNull ->
                json.addProperty("architecture", architectureNonNull)
            }
            brand?.let { brandNonNull ->
                json.addProperty("brand", brandNonNull)
            }
            model?.let { modelNonNull ->
                json.addProperty("model", modelNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Device {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Device {
                try {
                    val architecture = jsonObject.get("architecture")?.asString
                    val brand = jsonObject.get("brand")?.asString
                    val model = jsonObject.get("model")?.asString
                    return Device(architecture, brand, model)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }
        }
    }

    /**
     * OS properties
     * @param build Build of the OS
     * @param name Name of the OS
     * @param version Version of the OS
     */
    public data class Os(
        public val build: String? = null,
        public val name: String? = null,
        public val version: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            build?.let { buildNonNull ->
                json.addProperty("build", buildNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            version?.let { versionNonNull ->
                json.addProperty("version", versionNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Os {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Os {
                try {
                    val build = jsonObject.get("build")?.asString
                    val name = jsonObject.get("name")?.asString
                    val version = jsonObject.get("version")?.asString
                    return Os(build, name, version)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Os",
                        e
                    )
                }
            }
        }
    }

    /**
     * Error properties
     * @param stack The stack trace or the complementary information about the error
     * @param kind The error type or kind (or code in some cases)
     */
    public data class Error(
        public val stack: String? = null,
        public val kind: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            stack?.let { stackNonNull ->
                json.addProperty("stack", stackNonNull)
            }
            kind?.let { kindNonNull ->
                json.addProperty("kind", kindNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Error {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Error {
                try {
                    val stack = jsonObject.get("stack")?.asString
                    val kind = jsonObject.get("kind")?.asString
                    return Error(stack, kind)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                }
            }
        }
    }

    /**
     * The source of this event
     */
    public enum class Source(
        private val jsonValue: String,
    ) {
        ANDROID("android"),
        IOS("ios"),
        BROWSER("browser"),
        FLUTTER("flutter"),
        REACT_NATIVE("react-native"),
        UNITY("unity"),
        KOTLIN_MULTIPLATFORM("kotlin-multiplatform"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Source = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
